#include <boost/range/algorithm/count_if.hpp>
#include <boost/range/algorithm/find_if.hpp>
#include <internal/cache/role.h>

namespace sharpei {
namespace cache {

RoleCache& RoleCache::operator =(const RoleCache& other) {
    const auto locked = other.shards.shared_lock();
    *shards.unique_lock() = *locked;
    return *this;
}

RoleCache::OptRole RoleCache::getRole(Shard::Id shardId, const Address& address) const {
    const auto locked = shards.shared_lock();
    const auto shard = locked->find(shardId);
    if (shard == locked->end()) {
        return OptRole();
    }
    const auto host = shard->second.find(address);
    if (host == shard->second.end()) {
        return OptRole();
    }
    return host->second;
}

RoleCache::Hosts RoleCache::get(Shard::Id shardId) const {
    const auto locked = shards.shared_lock();
    const auto shard = locked->find(shardId);
    if (shard == locked->end()) {
        return RoleCache::Hosts();
    }
    return shard->second;
}

RoleCache::HostsSet RoleCache::get(Shard::Id shardId, Role role) const {
    const Hosts hosts = get(shardId);
    HostsSet res;
    for (const auto& addrWithRole: hosts) {
        if (addrWithRole.second == role) {
            res.push_back(addrWithRole.first);
        }
    }
    return res;
}

void RoleCache::update(Shard::Id shardId, const HostsWithOptRole& hosts) {
    const auto locked = shards.unique_lock();
    auto& oldHosts = (*locked)[shardId];
    updateOldMasters(oldHosts, hosts);
    oldHosts = makeNewHosts(oldHosts, hosts);
}

void RoleCache::erase(Shard::Id shardId) {
    shards.unique_lock()->erase(shardId);
}

RoleCache::Shards RoleCache::all() const {
    return *shards.shared_lock();
}

void RoleCache::updateOldMasters(Hosts& oldHosts, const HostsWithOptRole& updatedHosts) {
    const auto updatedMaster = boost::find_if(updatedHosts,
        [] (const auto& x) { return x.second && *x.second == Role::Master; });
    if (updatedMaster == updatedHosts.end()) {
        return;
    }
    for (auto& old : oldHosts) {
        const auto sameUpdated = updatedHosts.find(old.first);
        if (sameUpdated == updatedHosts.end() || !sameUpdated->second || sameUpdated->second.get() != Role::Master) {
            old.second = Role::Replica;
        }
    }
}

RoleCache::Hosts RoleCache::makeNewHosts(const Hosts& oldHosts, const HostsWithOptRole& updatedHosts) {
    Hosts newHosts;
    for (const auto& x : updatedHosts) {
        const auto& addr = x.first;
        const auto& optRole = x.second;
        if (optRole) {
            newHosts.insert({addr, *optRole});
        } else {
            const auto oldHost = oldHosts.find(addr);
            if (oldHost == oldHosts.end()) {
                newHosts.insert({addr, Role::Replica});
            } else {
                const auto oldRole = oldHost->second;
                newHosts.insert({addr, oldRole});
            }
        }
    }
    return newHosts;
}

} // namespace cache
} // namespace sharpei
