#include <boost/range/algorithm/find_if.hpp>
#include <internal/cache/state.h>

namespace sharpei {
namespace cache {

StateCache& StateCache::operator =(const StateCache& other) {
    const auto locked = other.shards.shared_lock();
    *shards.unique_lock() = *locked;
    return *this;
}

StateCache::OptState StateCache::get(Shard::Id shardId, const Address& address) const {
    const auto locked = shards.shared_lock();
    const auto shard = locked->find(shardId);
    if (shard == locked->end()) {
        return OptState();
    }
    const auto &hosts = shard->second;
    const auto host = hosts.find(address);
    if (host == hosts.end()) {
        return OptState();
    }
    return host->second;
}

void StateCache::erase(Shard::Id shardId) {
    shards.unique_lock()->erase(shardId);
}

void StateCache::update(Shard::Id shardId, const HostsWithOptState& hosts) {
    const auto locked = shards.unique_lock();
    auto& oldHosts = (*locked)[shardId];
    oldHosts = makeNewHosts(oldHosts, hosts);
}

StateCache::Hosts StateCache::makeNewHosts(const Hosts& oldHosts, const HostsWithOptState& updatedHosts) {
    Hosts newHosts;
    for (const auto& x : updatedHosts) {
        const auto& addr = x.first;
        const auto& optState = x.second;
        if (optState) {
            newHosts.insert({addr, *optState});
        } else {
            const auto oldHost = oldHosts.find(addr);
            if (oldHost == oldHosts.end()) {
                newHosts.insert({addr, {maxLag}});
            } else {
                const auto& oldState = oldHost->second;
                newHosts.insert({addr, oldState});
            }
        }
    }
    return newHosts;
}

} // namespace cache
} // namespace sharpei
