#include <boost/range/adaptor/transformed.hpp>
#include <internal/cache/status.h>

namespace sharpei {
namespace cache {

LimitedCounter::LimitedCounter(std::size_t max, std::size_t value)
        : max_(max), value_(value) {
    if (value > max) {
        throw std::logic_error("LimitedCounter::LimitedCounter: value > max");
    }
}

LimitedCounter& LimitedCounter::operator +=(std::size_t value) {
    value_ += std::min(value, max_ - value_);
    return *this;
}

LimitedCounter& LimitedCounter::operator -=(std::size_t value) {
    value_ -= std::min(value, value_);
    return *this;
}

std::size_t LimitedCounter::value() const {
    return value_;
}

Availability::Availability(std::size_t historyCapacity, std::size_t errorsLimit)
    : errorsLimit_(errorsLimit),
      errorsCount_(historyCapacity, 0),
      responseHistory_(historyCapacity) {}

Availability::Status Availability::get(GetMethod method) const {
    return (this->*method)();
}

Availability::Status Availability::smooth() const {
    if (responseHistory_.empty()) {
        throw std::logic_error("Availability::smooth: history is empty");
    }
    const auto lhs = errorsCount_.value() * responseHistory_.capacity();
    const auto rhs = errorsLimit_ * responseHistory_.size();
    return lhs < rhs ? Status::Alive : Status::Dead;
}

Availability::Status Availability::recent() const {
    if (responseHistory_.empty()) {
        throw std::logic_error("Availability::recent: history is empty");
    }
    return responseHistory_.back() ? Status::Alive : Status::Dead;
}

Availability::History Availability::getHistory() const {
    using boost::adaptors::transformed;
    const auto range = responseHistory_
        | transformed([] (bool x) { return x ? Status::Alive : Status::Dead; });
    return Availability::History(range.begin(), range.end());
}

void Availability::update(UpdateMethod method) {
    return (this->*method)();
}

void Availability::alive() {
    errorsCount_ -= responseHistory_.size() == responseHistory_.capacity()
                    && !responseHistory_.empty()
                    && !responseHistory_.front();
    responseHistory_.push_back(true);
}

void Availability::dead() {
    errorsCount_ += 1 - (responseHistory_.size() == responseHistory_.capacity()
                         && !responseHistory_.empty()
                         && !responseHistory_.front());
    responseHistory_.push_back(false);
}

StatusCache::StatusCache(std::size_t historyCapacity, std::size_t errorsLimit)
    : historyCapacity_(historyCapacity), errorsLimit_(errorsLimit) {}

std::size_t StatusCache::historyCapacity() const {
    return historyCapacity_;
}

std::size_t StatusCache::errorsLimit() const {
    return errorsLimit_;
}

StatusCache::OptStatus StatusCache::get(Shard::Id shardId, const Address& address, Availability::GetMethod method) const {
    const auto locked = shards.shared_lock();
    const auto shard = locked->find(shardId);
    if (shard == locked->end()) {
        return OptStatus();
    }
    const auto &hosts = shard->second;
    const auto host = hosts.find(address);
    if (host == hosts.end()) {
        return OptStatus();
    }
    return host->second.get(method);
}

StatusCache::History StatusCache::getHistory(Shard::Id shardId, const Address& address) const {
    const auto locked = shards.shared_lock();
    const auto shard = locked->find(shardId);
    if (shard == locked->end()) {
        return History();
    }
    const auto &hosts = shard->second;
    const auto host = hosts.find(address);
    if (host == hosts.end()) {
        return History();
    }
    return host->second.getHistory();
}

void StatusCache::alive(Shard::Id shardId, const Address& address) {
    update(shardId, address, &Availability::alive);
}

void StatusCache::dead(Shard::Id shardId, const Address& address) {
    update(shardId, address, &Availability::dead);
}

void StatusCache::update(Shard::Id shardId, const Address& address, Availability::UpdateMethod method) {
    const auto locked = shards.unique_lock();
    auto& hosts = (*locked)[shardId];
    auto host = hosts.find(address);
    if (host == hosts.end()) {
        host = hosts.insert({address, Availability(historyCapacity_, errorsLimit_)}).first;
    }
    host->second.update(method);
}

void StatusCache::erase(Shard::Id shardId) {
    shards.unique_lock()->erase(shardId);
}

} // namespace cache
} // namespace sharpei
