#include <pa/async.h>
#include <internal/logger.h>
#include <internal/stats.h>
#include <internal/services/iam/iam_client.h>
#include <internal/services/iam/iam_token_cache.h>
#include <internal/services/iam/json_web_token_file_reader.h>
#include <internal/services/yc/yc_client.h>
#include <internal/services/yc/yc_hosts_cache.h>
#include <internal/services/yc/reflection.h>
#include <internal/cloud/server.h>
#include <internal/cloud/module.h>

#include <yamail/data/deserialization/json_reader.h>

#include <ymod_httpclient/cluster_client.h>

#include <yplatform/module_registration.h>

#include <cstdlib>

namespace sharpei::cloud {

void Module::init(const yplatform::ptree& data) {
    using sharpei::cache::Cache;
    using services::iam::IamClient;
    using services::iam::IamTokenCache;
    using services::iam::JsonWebTokenFileReader;
    using services::yc::YcClient;
    using namespace std::literals;

    config = makeConfig(data);

    if (const auto dbname = std::getenv("SHARPEI_CLUSTER_DBNAME")) {
        config.poller.shardsProvider.dbname = dbname;
        LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message="use cluster dbname from env SHARPEI_CLUSTER_DBNAME: \""s
            + config.poller.shardsProvider.dbname + "\"");
    } else {
        LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message="SHARPEI_CLUSTER_DBNAME env is not set, use value from config: \""s
            + config.poller.shardsProvider.dbname + "\"");
    }

    if (const auto user = std::getenv("SHARPEI_CLUSTER_USER")) {
        const std::string none = "";
        config.shard.adaptor.authInfo.user(user);
        LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message="use cluster user from env SHARPEI_CLUSTER_USER: \""
            + (config.shard.adaptor.authInfo.user() ? *config.shard.adaptor.authInfo.user() : none) + "\"");
    } else {
        const std::string none = "";
        LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message="SHARPEI_CLUSTER_USER env is not set, use value from config: \""
            + (config.shard.adaptor.authInfo.user() ? *config.shard.adaptor.authInfo.user() : none) + "\"");
    }

    if (const auto clusterId = std::getenv("SHARPEI_CLUSTER_ID")) {
        config.yc.client.clusterId = clusterId;
        LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message="use cluster id from env SHARPEI_CLUSTER_ID: \""
            + config.yc.client.clusterId + "\"");
    } else {
        LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message="SHARPEI_CLUSTER_ID env is not set, use value from config: \""
            + config.yc.client.clusterId + "\"");
    }

    if (!pa::async_profiler::is_init()) {
        pa::async_profiler::init(1000000, 16000, config.pa.log);
    }

    cache = std::make_shared<Cache>(config.cache.historyCapacity, config.cache.errorsLimit);
    shardPool = db::makeConnectionPool(config.shard.pool, io);

    uniqId = generateUniqId();

    const auto iamTokenCache = std::make_shared<IamTokenCache>();
    const auto hostsCache = std::make_shared<YcHostsCache>(readFsHostsCache(config.yc.poller.fsHostsCachePath));


    const auto iamHttpClientConfig = data.get_child("iam.client.configuration");

    const auto iamHttpClient = std::make_shared<yhttp::cluster_client>(io, iamHttpClientConfig);
    iamPoller = std::make_shared<IamPoller>(
        uniqId,
        config.iam.poller,
        std::make_shared<IamClient>(
            config.iam.client,
            iamHttpClient,
            std::make_shared<JsonWebTokenFileReader>(config.iam.jwtFilePath)
        ),
        iamTokenCache
    );

    const auto ycHttpClientConfig = data.get_child("yc.client.configuration");

    const auto ycHttpClient = std::make_shared<yhttp::cluster_client>(io, ycHttpClientConfig);
    ycPoller = std::make_shared<YcPoller>(
        uniqId,
        config.yc.poller,
        std::make_shared<YcClient>(config.yc.client, ycHttpClient),
        iamTokenCache,
        hostsCache
    );

    poller = std::make_unique<ClusterPoller>(config.poller, config.shard, shardPool, hostsCache, cache);

    server::bindHandlers(cache);

    LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message="module initialized");
}

void Module::start() {
    worker = std::make_unique<IoThreadWorker>(uniqId, io);

    iamPoller->start(io);
    ycPoller->start(io);
    poller->start(io);

    LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message="module started");
}

void Module::stop() {
    poller->stop();
    ycPoller->stop();
    iamPoller->stop();
    worker->stop();
    io.stop();
    worker.reset();

    using namespace std::literals;
    LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message="module stopped");
}

void Module::fini() {
    iamPoller.reset();
    ycPoller.reset();
    poller.reset();

    using namespace std::literals;
    LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message="module finished");
}

yplatform::ptree Module::get_stats() const {
    yplatform::ptree result;
    stats::addShardPool(shardPool, result);
    return result;
}

YcHostsCache Module::readFsHostsCache(const std::string& path) const {
    using yamail::data::deserialization::fromJson;
    using namespace std::literals;

    std::ifstream file(path);

    if (!file.is_open()) {
        LOGDOG_(log::GetLogger(log::sharpeiLogKey), warning, log::message="can't open file with hosts cache: \""s + path + "\"");
        return {};
    }

    std::noskipws(file);

    const std::string data {std::istreambuf_iterator<char>(file), std::istreambuf_iterator<char>()};

    if (file.bad() || file.fail()) {
        LOGDOG_(log::GetLogger(log::sharpeiLogKey), warning, log::message="read hosts cache from file error: \"" + path + "\"");
        return {};
    }

    try {
        return fromJson<YcHostsCache::value_type>(data);
    } catch (const std::exception& e) {
        LOGDOG_(log::GetLogger(log::sharpeiLogKey), warning, log::message="failed to parse value from file \"" + path + "\"", log::exception=e);
        return {};
    }
}

} // namespace sharpei::cloud

DEFINE_SERVICE_OBJECT(sharpei::cloud::Module)
