#include <unordered_map>
#include <yplatform/find.h>
#include <apq/connection.hpp>
#include <internal/errors.h>
#include <internal/db/conn_info.h>
#include <internal/db/adaptors/meta_adaptor.h>
#include <boost/range/algorithm/transform.hpp>

namespace sharpei {
namespace db {
namespace detail {

class GetShardAddrsExecutor: public RequestExecutor
                           , public EnableStaticConstructor<GetShardAddrsExecutor> {
private:
    friend class EnableStaticConstructor<GetShardAddrsExecutor>;

    GetShardAddrsExecutor(const AdaptorConfig& config, const MetaPoolPtr& pool, const Scribe& scribe,
                          Shard::Id shardId, const GetShardHandler& handler, const ErrorHandler& errorHandler)
        : RequestExecutor(config, pool, scribe, errorHandler)
        , shardId_(shardId)
        , handler_(handler)
    {}

    apq::query makeQuery() const final {
        static constexpr char queryStr[] =
            "SELECT s.name AS shard_name, shard_id, host, port, dbname, dc "
              "FROM shards.instances "
              "JOIN shards.shards AS s USING (shard_id) "
             "WHERE shard_id = $1";
        apq::query query(queryStr);
        query.bind_const_int64(shardId_);
        return query;
    }

    std::string getProfilingId() const final {
        return "get_shard";
    }

    void onEmptyResult() final {
        errorHandler_(ExplainedError(Error::shardNotFound,
            makeErrorMessage(*query_, "shard " + std::to_string(shardId_) + " not found")));
    }

    void onCorrectResult(apq::row_iterator it) final {
        try {
            const auto shardName = extractField<Shard::Name>(it, "shard_name");
            ShardWithoutRoles::Addresses addrs;
            for (; it != apq::row_iterator(); ++it) {
                const auto addr = getAddrFromPgResult(it);
                addrs.push_back(addr);
                LOGDOG_(scribe_.logger, debug, log::shard_id=shardId_, log::host=addr.host, log::port=addr.port);
            }
            handler_(ShardWithoutRoles(shardId_, shardName, std::move(addrs)));
        } catch (const std::exception& e) {
            errorHandler_(ExplainedError(Error::metaRequestError, makeErrorMessage(*query_, e.what())));
        }
    }

    Shard::Id shardId_;
    GetShardHandler handler_;
};

class GetAllShardsExecutor: public RequestExecutor
                          , public EnableStaticConstructor<GetAllShardsExecutor> {
private:
    friend class EnableStaticConstructor<GetAllShardsExecutor>;

    GetAllShardsExecutor(const AdaptorConfig& config, const MetaPoolPtr& pool, const Scribe& scribe,
                         const GetAllShardsHandler& handler, const ErrorHandler& errorHandler)
        : RequestExecutor(config, pool, scribe, errorHandler)
        , handler_(handler)
    {}

    apq::query makeQuery() const final {
        static constexpr char queryStr[] =
            "SELECT s.name AS shard_name, shard_id, host, port, dbname, dc "
              "FROM shards.instances "
              "JOIN shards.shards AS s USING (shard_id)";
        apq::query query(queryStr);
        return query;
    }

    std::string getProfilingId() const final {
        return "get_all_shards";
    }

    void onEmptyResult() final {
        errorHandler_(ExplainedError(Error::metaRequestError, makeErrorMessage(*query_, "no result")));
    }

    void onCorrectResult(apq::row_iterator it) final {
        try {
            std::unordered_map<Shard::Id, ShardWithoutRoles::Addresses> addrs;
            std::unordered_map<Shard::Id, Shard::Name> names;
            for (; it != apq::row_iterator(); ++it) {
                const auto shardId = extractField<Shard::Id>(it, "shard_id");
                const auto shardName = extractField<Shard::Name>(it, "shard_name");
                const auto addr = getAddrFromPgResult(it);
                addrs[shardId].push_back(addr);
                names.emplace(shardId, shardName);
                LOGDOG_(scribe_.logger, debug, log::shard_id=shardId, log::host=addr.host, log::port=addr.port);
            }
            std::vector<ShardWithoutRoles> shards;
            shards.reserve(addrs.size());
            boost::transform(addrs, std::back_inserter(shards), [&] (auto& v) {
                return ShardWithoutRoles(v.first, names[v.first], std::move(v.second));
            });
            handler_(std::move(shards));
        } catch (const std::exception& e) {
            errorHandler_(ExplainedError(Error::metaRequestError, makeErrorMessage(*query_, e.what())));
        }
    }

    GetAllShardsHandler handler_;
};

class PingExecutor: public RequestExecutor
                  , public EnableStaticConstructor<PingExecutor> {
private:
    friend class EnableStaticConstructor<PingExecutor>;

    PingExecutor(const AdaptorConfig& config, const MetaPoolPtr& pool, const Scribe& scribe,
                 const FinishHandler& handler, const ErrorHandler& errorHandler)
        : RequestExecutor(config, pool, scribe, errorHandler)
        , handler_(handler)
    {}

    apq::query makeQuery() const final {
        static apq::query query("SELECT 1");
        return query;
    }

    std::string getProfilingId() const final {
        return "ping";
    }

    void onEmptyResult() final {
        errorHandler_(ExplainedError(Error::metaRequestError, makeErrorMessage(*query_, "no result")));
    }

    void onCorrectResult(apq::row_iterator it) final {
        try {
            std::int32_t result = 0;
            it->at(0, result);

            if (result == 1) {
                handler_();
            } else {
                errorHandler_(ExplainedError(Error::metaRequestError,
                    makeErrorMessage(*query_, "result differs from expected: " + std::to_string(result))));
            }
        } catch (const std::exception& e) {
            errorHandler_(ExplainedError(Error::metaRequestError, makeErrorMessage(*query_, e.what())));
        }
    }

    FinishHandler handler_;
};

class GetDomainShardIdExecutor : public RequestExecutor
                               , public EnableStaticConstructor<GetDomainShardIdExecutor> {
private:
    friend class EnableStaticConstructor<GetDomainShardIdExecutor>;

    GetDomainShardIdExecutor(const AdaptorConfig& config, const MetaPoolPtr& pool, const Scribe& scribe,
            const DomainId domainId, const GetShardIdHandler& handler, const ErrorHandler& errorHandler)
        : RequestExecutor(config, pool, scribe, errorHandler)
        , domainId(domainId)
        , handler(handler)
    {}

    apq::query makeQuery() const final {
        apq::query result("SELECT shard_id FROM shards.domains_organizations WHERE domain_id = $1::bigint");
        result.bind_const_int64(domainId);
        return result;
    }

    std::string getProfilingId() const final {
        return "get_domain_shard_id";
    }

    void onEmptyResult() final {
        errorHandler_(ExplainedError(Error::domainNotFound, "no result"));
    }

    void onCorrectResult(apq::row_iterator it) final {
        try {
            handler(extractField<Shard::Id>(it, "shard_id"));
        } catch (const std::exception& e) {
            errorHandler_(ExplainedError(Error::metaRequestError, makeErrorMessage(*query_, e.what())));
        }
    }

    DomainId domainId;
    GetShardIdHandler handler;
};

class GetOrganizationShardIdExecutor : public RequestExecutor
                                     , public EnableStaticConstructor<GetOrganizationShardIdExecutor> {
private:
    friend class EnableStaticConstructor<GetOrganizationShardIdExecutor>;

    GetOrganizationShardIdExecutor(const AdaptorConfig& config, const MetaPoolPtr& pool, const Scribe& scribe,
            const OrgId orgId, const GetShardIdHandler& handler, const ErrorHandler& errorHandler)
        : RequestExecutor(config, pool, scribe, errorHandler)
        , orgId(orgId)
        , handler(handler)
    {}

    apq::query makeQuery() const final {
        apq::query result("SELECT shard_id FROM shards.domains_organizations WHERE org_id = $1::bigint");
        result.bind_const_int64(orgId);
        return result;
    }

    std::string getProfilingId() const final {
        return "get_org_shard_id";
    }

    void onEmptyResult() final {
        errorHandler_(ExplainedError(Error::organizationNotFound, "no result"));
    }

    void onCorrectResult(apq::row_iterator it) final {
        try {
            handler(extractField<Shard::Id>(it, "shard_id"));
        } catch (const std::exception& e) {
            errorHandler_(ExplainedError(Error::metaRequestError, makeErrorMessage(*query_, e.what())));
        }
    }

    OrgId orgId;
    GetShardIdHandler handler;
};

BaseMetaAdaptorImpl::BaseMetaAdaptorImpl(const AdaptorConfig& config,
                                         const MetaPoolPtr& pool,
                                         const Scribe& scribe,
                                         MetaMasterProviderPtr metaMasterProvider)
    : config_(config),
      pool_(pool),
      scribe_(scribe),
      metaMasterProvider_(std::move(metaMasterProvider)) {
}

void BaseMetaAdaptorImpl::getShard(Shard::Id shardId, const GetShardHandler& handler, const ErrorHandler& errorHandler) const {
    auto executor = GetShardAddrsExecutor::create(config_, pool_, scribe_, shardId, handler, errorHandler);
    executor->execute();
}

void BaseMetaAdaptorImpl::getAllShards(const GetAllShardsHandler& handler, const ErrorHandler& errorHandler) const {
    auto executor = GetAllShardsExecutor::create(config_, pool_, scribe_, handler, errorHandler);
    executor->execute();
}

void BaseMetaAdaptorImpl::ping(const FinishHandler& handler, const ErrorHandler& errorHandler) const {
    auto executor = PingExecutor::create(config_, pool_, scribe_, handler, errorHandler);
    executor->execute();
}

void BaseMetaAdaptorImpl::getMaster(const GetMasterHandler& handler,
                                    const ErrorHandler& errorHandler) const {
    metaMasterProvider_->getMaster(handler, errorHandler);
}

void BaseMetaAdaptorImpl::getRegData(GetRegDataHandler handler, ErrorHandler errorHandler) const {
    auto executor = GetRegDataExecutor<std::monostate>::create(
        config_, pool_, scribe_, std::monostate{}, std::move(handler), std::move(errorHandler));
    executor->execute();
}

void BaseMetaAdaptorImpl::getDomainShardId(const DomainId domainId, GetShardIdHandler handler,
        ErrorHandler errorHandler) const {
    auto executor = GetDomainShardIdExecutor::create(config_, pool_, scribe_, domainId,
        std::move(handler), std::move(errorHandler));
    executor->execute();
}

void BaseMetaAdaptorImpl::getOrganizationShardId(const OrgId orgId, GetShardIdHandler handler,
        ErrorHandler errorHandler) const {
    auto executor = GetOrganizationShardIdExecutor::create(config_, pool_, scribe_, orgId,
        std::move(handler), std::move(errorHandler));
    executor->execute();
}

} // namespace detail
} // namespace db
} // namespace sharpei
