#include <internal/db/adaptors/request_executor.h>

#include <internal/errors.h>
#include <internal/helpers.h>

namespace sharpei::db {

void RequestExecutor::execute() {
    query_.reset(new apq::query(makeQuery()));
    auto handler = [self = shared_from_this()](const apq::result& res, apq::row_iterator it) {
        self->handleRequest(res, std::move(it));
    };
    const auto timeout = config_.requestTimeout;
    auto profilingCtx = std::make_pair(scribe_.profiler, getProfilingId());
    pool_->async_request(
        *query_, std::move(handler), apq::result_format_binary, timeout, std::move(profilingCtx));
}

RequestExecutor::RequestExecutor(const AdaptorConfig& config,
                                 const MetaPoolPtr& pool,
                                 const Scribe& scribe,
                                 const ErrorHandler& errorHandler)
    : AsyncExecutor(config, pool, scribe, errorHandler) {
}

Shard::Database::Address RequestExecutor::getAddrFromPgResult(apq::row_iterator it) {
    const std::string host = extractField<std::string>(it, "host");
    const auto port = extractField<unsigned>(it, "port");
    const std::string dbname = extractField<std::string>(it, "dbname");
    const std::string dataCenter = extractField<std::string>(it, "dc");
    return Shard::Database::Address(host, port, dbname, dataCenter);
}

void RequestExecutor::handleRequest(const apq::result& res, apq::row_iterator it) {
    if (res.code()) {
        errorHandler_(ExplainedError(chooseMetaRequestErrorCode(res),
                                     makeErrorMessage(*query_, res.message())));
        return;
    }

    if (it == apq::row_iterator()) {
        onEmptyResult();
        return;
    }

    onCorrectResult(it);
}

}  // namespace sharpei::db
