#include <boost/algorithm/string/join.hpp>
#include <boost/asio/steady_timer.hpp>
#include <yplatform/find.h>
#include <apq/connection.hpp>
#include <internal/helpers.h>
#include <internal/async_profile.h>
#include <internal/db/conn_info.h>
#include <internal/db/adaptors/shard_adaptor.h>
#include <internal/db/pools/connection_pool.h>
#include <boost/range/algorithm/count_if.hpp>

namespace sharpei {
namespace db {

class ShardAdaptorImpl: public ShardAdaptor {
public:
    ShardAdaptorImpl(const ShardConfig::AdaptorConfig& config, const ShardPoolPtr& pool,
                     const Scribe& scribe, const cache::CachePtr& cache)
        : config_(config), pool_(pool), scribe_(scribe), cache_(cache)
    {}

    void resetHostCache(const ShardWithoutRoles& shard, const FinishHandler& handler, const ErrorHandler& errorHandler) override {
        auto executor = ResetHostCacheExecutor::create(config_, pool_, scribe_, cache_,
                                                       shard, handler, errorHandler);
        executor->execute();
    }

private:
    class AsyncExecutor: public std::enable_shared_from_this<AsyncExecutor> {
    public:
        virtual ~AsyncExecutor() {}

        void execute() {
            auto ptr = shared_from_this();
            for (const auto& addr: shard_.addrs) {
                const auto connInfo = ConnectionInfo(addr.host, addr.port, addr.dbname, config_.authInfo);
                const auto connection = pool_->get(connInfo);

                auto handler = std::bind(&AsyncExecutor::handleRequest, ptr, std::placeholders::_1, std::placeholders::_2, addr);
                const auto timeout = config_.requestTimeout;
                Profile profile(scribe_.profiler, "get_status", addr.host);
                apq::connection_pool::request_handler_t profiledHandler = asyncProfile(std::move(profile), std::move(handler));

                connection->async_request(queryIsReplica_, profiledHandler, apq::result_format_binary, timeout);
            }
        }

    protected:
        using RoleCache = cache::RoleCache;
        using StateCache = cache::StateCache;

        virtual void onRequestError(const Shard::Database::Address& addr) = 0;
        virtual void onCorrectResult(apq::row_iterator it, const Shard::Database::Address& addr) = 0;

        AsyncExecutor(const ShardConfig::AdaptorConfig& config, const ShardPoolPtr& pool, const Scribe& scribe,
                      const cache::CachePtr& cache, const ShardWithoutRoles& shard, const ErrorHandler& errorHandler)
            : config_(config), pool_(pool), scribe_(scribe), cache_(cache)
            , shard_(shard)
            , errorHandler_(errorHandler)
        {}

        Shard::Database::Role getRoleFromPgResult(apq::row_iterator it) {
            bool isInRecovery = true;
            try {
                it->at("is_in_recovery", isInRecovery);
            } catch (const std::exception& e) {
                throw PgException(std::string("can't extract field is_in_recovery from pg result: ") + e.what());
            }
            return isInRecovery ? Shard::Database::Role::Replica : Shard::Database::Role::Master;
        }

        Shard::Database::State getStateFromPgResult(apq::row_iterator it) {
            Shard::Database::State state {0};
            try {
                it->at("replication_lag", state.lag);
            } catch (const std::exception& e) {
                throw PgException(std::string("can't extract field replication_lag from pg result: ") + e.what());
            }
            return state;
        }

        void addAliveHost(const Shard::Database::Address& address) {
            cache_->status.alive(shard_.id, address);
        }

        void addDeadHost(const Shard::Database::Address& address) {
            cache_->status.dead(shard_.id, address);
        }

        void addHostWithRole(const Shard::Database::Address& address, const RoleCache::OptRole& role) {
            std::lock_guard<Spinlock> guard(hostsLock_);
            hosts_.insert({address, role});
        }

        void addHostWithState(const Shard::Database::Address& address, const StateCache::OptState& state) {
            std::lock_guard<Spinlock> guard(hostsLock_);
            hostsState_.insert({address, state});
        }

        void resetCache() {
            cache_->shardName.update(shard_.id, shard_.name);
            cache_->role.update(shard_.id, hosts_);
            cache_->state.update(shard_.id, hostsState_);
            logStatus();
        }

        ShardConfig::AdaptorConfig config_;
        ShardPoolPtr pool_;
        Scribe scribe_;
        cache::CachePtr cache_;
        ShardWithoutRoles shard_;
        ErrorHandler errorHandler_;
        RoleCache::HostsWithOptRole hosts_;
        StateCache::HostsWithOptState hostsState_;
        Spinlock hostsLock_;

    private:
        void handleRequest(const apq::result& res, apq::row_iterator it, const Shard::Database::Address& addr) {
            if (res.code()) {
                using namespace std::literals;
                LOGDOG_(scribe_.logger, error, log::host=addr.host, log::port=addr.port,
                    log::message="error in request to shard host: "s + res.message(), log::apq_result=res);
                onRequestError(addr);
                return;
            }

            if (it == apq::row_iterator()) {
                LOGDOG_(scribe_.logger, error, log::host=addr.host, log::port=addr.port,
                    log::message="error in request to shard host: no result in response", log::apq_result=res);
                onRequestError(addr);
                return;
            }

            onCorrectResult(it, addr);
        }

        void logStatus() const {
            using cache::Availability;
            for (const auto& host : hosts_) {
                const auto addr = host.first;
                const auto role = cache_->role.getRole(shard_.id, addr);
                const auto state = cache_->state.get(shard_.id, addr);
                const auto smoothStatus = cache_->status.get(shard_.id, addr, &Availability::smooth);
                const auto statusHistory = cache_->status.getHistory(shard_.id, addr);
                std::vector<std::string> states;
                for (auto status : statusHistory) {
                    states.push_back(status.toString());
                }
                using namespace log;
                using namespace std::literals;
                auto hostRole = role ? role->toString() : "unknown";
                auto hostLag = state ? std::to_string(state->lag) : "unknown";
                auto smoothState = smoothStatus ? smoothStatus->toString() : "unknown";
                auto hostStates = "["s + boost::join(states, ", ") + "]";
                LOGDOG_(scribe_.status, notice, shard_id=shard_.id, shard_name=shard_.name, log::host=addr.host,
                    port=addr.port, log::role=hostRole, lag=hostLag, smooth_state=smoothState, log::states=hostStates);
            }
        }

        static const apq::query queryIsReplica_;
    };

    class ResetHostCacheExecutor: public AsyncExecutor
                                , public EnableStaticConstructor<ResetHostCacheExecutor> {
    public:
        ~ResetHostCacheExecutor() {
            try {
                resetCache();
                if (errorHosts_.empty()) {
                    handler_();
                } else {
                    errorHandler_(ExplainedError(Error::resetError,
                        "could not reset info for hosts: " + boost::join(errorHosts_, ",")));
                }
            } catch (const std::exception& e) {
                errorHandler_(ExplainedError(Error::resetError,
                    "could not reset info: " + std::string(e.what())));
            }
        }

    private:
        friend class EnableStaticConstructor<ResetHostCacheExecutor>;

        ResetHostCacheExecutor(const ShardConfig::AdaptorConfig& config, const ShardPoolPtr& pool, const Scribe& scribe,
                               const cache::CachePtr& cache, const ShardWithoutRoles& shard,
                               const FinishHandler& handler, const ErrorHandler& errorHandler)
            : AsyncExecutor(config, pool, scribe, cache, shard, errorHandler)
            , handler_(handler)
        {}

        void onRequestError(const Shard::Database::Address& addr) override {
            rememberErrorHost(addr);
            addDeadHost(addr);
            addHostWithRole(addr, RoleCache::OptRole());
            addHostWithState(addr, StateCache::OptState());
        }

        void onCorrectResult(apq::row_iterator it, const Shard::Database::Address& addr) override {
            try {
                const auto role = getRoleFromPgResult(it);
                const auto state = (role == Shard::Database::Role::Master ?
                        Shard::Database::State{0} : getStateFromPgResult(it));

                addAliveHost(addr);
                addHostWithRole(addr, role);
                addHostWithState(addr, state);

                if (++it != apq::row_iterator()) {
                    LOGDOG_(scribe_.logger, warning, log::host=addr.host, log::port=addr.port, log::message="more than one row returned for getting host info");
                }
            } catch (std::exception& e) {
                LOGDOG_(scribe_.logger, error, log::message="error in request to shard host", log::host=addr.host, log::port=addr.port, log::exception=e);
                rememberErrorHost(addr);
                return;
            }
        }

        void rememberErrorHost(const Shard::Database::Address& addr) {
            std::lock_guard<Spinlock> guard(errorHostsLock_);
            errorHosts_.push_back(addr.host);
        }

        FinishHandler handler_;
        std::vector<std::string> errorHosts_;
        Spinlock errorHostsLock_;
    };

    ShardConfig::AdaptorConfig config_;
    ShardPoolPtr pool_;
    Scribe scribe_;
    cache::CachePtr cache_;
};


const apq::query ShardAdaptorImpl::AsyncExecutor::queryIsReplica_(
        "SELECT pg_is_in_recovery() as is_in_recovery, "
        "ROUND(extract(epoch FROM clock_timestamp() - pg_last_xact_replay_timestamp()))::integer as replication_lag;"
);

ShardAdaptorPtr getShardAdaptor(const ShardConfig::AdaptorConfig& config, const ShardPoolPtr& pool,
                                const Scribe& scribe, const cache::CachePtr& cache) {
    return std::make_shared<ShardAdaptorImpl>(config, pool, scribe, cache);
}

} // namespace db
} // namespace sharpei
