#include <mutex>
#include <string_view>

#include <yplatform/module_registration.h>

#include <internal/logger.h>
#include <internal/mail/module.h>
#include <internal/mail/server/server.h>

#include <user_journal/service_factory.h>

namespace sharpei {
namespace mail {

void Module::init(const yplatform::ptree& data) {
    Base::init(data);

    config = makeConfig(getConfig(), data);

    auto ujLogger = std::make_shared<yplatform::log::source>(YGLOBAL_LOG_SERVICE, "user_journal");
    const auto writer = [ujLogger = std::move(ujLogger)](std::string_view line) {
        if (line.empty()) {
            return;
        }

        if (line.ends_with("\n")) {
            line.remove_suffix(1);
        }
        
        YLOG(*ujLogger, info) << line;
    };

    userJournalService = ::user_journal::ServiceFactory()
            .tskv(data.get_child("user_journal"), writer)
            .logger(boost::make_shared<UserJournalLog>()).product();

    server::bindHandlers(config,
                         getShardsCache(),
                         getMetaCache(),
                         getMetaPool(),
                         getShardPool(),
                         getPeersPool(),
                         getMetaAdaptorFactory(),
                         userJournalService);

    LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message="module initialized");
}

void Module::reload(const yplatform::ptree&) {
    LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message="module reloaded");
}

} // namespace mail
} // namespace sharpei

DEFINE_SERVICE_OBJECT(sharpei::mail::Module)
