#ifndef SHARPEI_MAIL_LAZY_REGISTRATION_H
#define SHARPEI_MAIL_LAZY_REGISTRATION_H

#include <internal/mail/registration.h>
#include <internal/random.h>
#include <internal/weighted_shuffle.h>
#include <boost/range/algorithm/find_if.hpp>

namespace sharpei {
namespace mail {
namespace registration {

Performer::Performer(std::shared_ptr<ymod_httpclient::cluster_call> httpClient,
        MetaAdaptorPtr metaAdaptor,
        cache::CachePtr cache,
        RegAdaptorPtr regAdaptor,
        ConfigPtr config,
        UserJournalPtr journal)
    : httpClient_(std::move(httpClient)),
      metaAdaptor_(std::move(metaAdaptor)),
      cache_(std::move(cache)),
      regAdaptor_(std::move(regAdaptor)),
      config_(std::move(config)),
      journal_(std::move(journal)){
}

Performer::ShardsMasters Performer::getRandomShardsIdsWithAliveMaster(WeightedShardIds& weightedShardIds) const {
    using boost::adaptors::transformed;

    const auto shuffled = weightedShuffle(weightedShardIds.begin(), weightedShardIds.end(), threadLocalGenerator());
    std::vector<std::pair<Shard::Id, Shard::Database::Address>> result;
    result.reserve(shuffled.size());
    for (auto index : shuffled) {
        const auto shardId = weightedShardIds[index].first;
        auto address = cache_->getAliveShardMaster(shardId);
        if (!address.is_initialized()) {
            continue;
        }
        result.emplace_back(shardId, std::move(address.get()));
    }
    return result;
}

Performer::ShardsMasters Performer::getAliveShardMaster(Shard::Id shardId) const {
    auto address = cache_->getAliveShardMaster(shardId);
    if (!address.is_initialized()) {
        return {};
    }
    return {{shardId, std::move(address.get())}};
}

void Performer::logRegistration(const ::user_journal::parameters::id::RegParams &params) const {
    if (journal_) {
        using namespace ::user_journal::parameters;
        journal_->write<Registration>(
                    id::hidden(false), id::mdb("pg"),
                    id::affected(1), id::state(""),
                    id::RegParams(params));
    }
}

namespace detail {

std::pair<ExplainedError, UserInfo> parseUserInfo(const yhttp::response& userInfo) {
    try {
        if (userInfo.status != 200) {
            std::string error = fmt::format("http status is {}", userInfo.status);
            return {ExplainedError(Error::blackBoxHttpError, error), UserInfo {}};
        }
        const auto infoResponse = bb::InfoResponse(userInfo.body);
        if (bb::Uid(infoResponse.get()).uid().empty()) {
            return {ExplainedError(Error::uidNotFound, "empty uid in blackbox result"), UserInfo {}};
        }
        const bb::DBFields dbFields(infoResponse.get());
        const auto suid = dbFields.get("subscription.suid.2");
        if (suid.empty()) {
            return {ExplainedError(Error::uidNotFound, "empty subscription.suid.2 in blackbox result"), UserInfo {}};
        }
        const auto dbId = dbFields.get("hosts.db_id.2");
        if (dbId != "pg") {
            return {ExplainedError(Error::notPgUser, "hosts.db_id.2 is not pg in blackbox result: '" + dbId + "'"), UserInfo {}};
        }
        const bb::PDDUserInfo pddUserInfo(infoResponse.get());
        const auto& domainId = pddUserInfo.domId();
        DomainId numericDomainId = 0;
        if (!domainId.empty()) {
            bool ok = false;
            std::tie(ok, numericDomainId) = lexicalCast<DomainId>(domainId);
            if (!ok) {
                return {ExplainedError(Error::blackBoxParseError, "parse domain id error"), UserInfo {}};
            }
        }
        const bb::AliasList aliases(infoResponse.get());
        const auto hasMailishAlias = aliases.getAliases().end() != boost::find_if(aliases.getAliases(),
            [] (const auto& v) { return v.type() == bb::AliasList::Item::Mailish; });
        UserInfo result {
            dbFields.get("account_info.country.uid"),
            dbFields.get("userinfo.lang.uid"),
            numericDomainId,
            hasMailishAlias
        };
        return {ExplainedError(Error::ok), std::move(result)};
    } catch (const bb::TempError& exception) {
        return {ExplainedError(Error::blackBoxTempError, exception.what()), UserInfo {}};
    } catch (const bb::FatalError& exception) {
        return {ExplainedError(Error::blackBoxFatalError, exception.what()), UserInfo {}};
    } catch (const boost::exception& exception) {
        std::string error = fmt::format("{} in {} at {}:{}",
            boost::diagnostic_information(exception), __PRETTY_FUNCTION__, __FILE__, __LINE__);
        return {ExplainedError(Error::blackBoxParseError, error), UserInfo {}};
    } catch (const std::exception& exception) {
        std::string error = fmt::format("{} in {} at {}:{}",
            exception.what(), __PRETTY_FUNCTION__, __FILE__, __LINE__);
        return {ExplainedError(Error::blackBoxParseError, error), UserInfo {}};
    }
}

bool isPddUser(const UserInfo& userInfo) {
    return userInfo.domainId != 0;
}

} // namespace detail
} // namespace registration
} // namespace mail
} // namespace sharpei

#endif // SHARPEI_MAIL_LAZY_REGISTRATION_H
