#include <internal/cache/cache.h>
#include <internal/config.h>
#include <internal/db/adaptors/meta_adaptor_factory.h>
#include <internal/db/adaptors/meta_cache_based_meta_master_provider.h>
#include <internal/db/ep/meta_cache_based_endpoint_provider.h>
#include <internal/db/ep/single_endpoint_provider.h>
#include <internal/db/pools/meta_pool.h>
#include <internal/logger.h>
#include <internal/module.h>
#include <internal/poller/meta_shards_provider.h>
#include <internal/stats.h>

#include <pa/async.h>
#include <yplatform/module_registration.h>

#include <memory>
#include <mutex>

using namespace sharpei;

namespace {

db::EndpointProviderPtr getEndpointProvider(ConfigPtr config, cache::CachePtr metaCache) {
    if (config->metaPoller.enabled) {
        const auto& authInfo = config->meta.db.adaptor.connInfoWOHost.authInfo;
        const auto& dcVanga = config->meta.endpointProvider.dcVanga;
        const auto& aliveHostsThreshold = config->meta.endpointProvider.aliveHostsThreshold;
        LOGDOG_(
            log::GetLogger(log::sharpeiLogKey), notice, log::message = "MetaCacheBasedEndpointProvider will be used");
        return makeMetaCacheBasedEndpointProvider(std::move(metaCache), authInfo, dcVanga, aliveHostsThreshold);
    } else {
        LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message = "SingleEndpointProvider will be used");
        return db::makeSingleEndpointProvider(config->meta);
    }
}

class ReplMonBasedMetaAdaptorFactory : public sharpei::db::IMetaAdaptorFactory {
private:
    sharpei::db::MetaMasterProviderPtr getMetaMasterProvider(const db::AdaptorConfig& config,
                                                             const db::MetaPoolPtr& pool,
                                                             const Scribe& scribe) const override {
        return std::make_shared<db::ReplMonBasedMetaMasterProvider>(config, pool, scribe);
    }
};

class MetaCacheBasedMetaAdaptorFactory : public db::IMetaAdaptorFactory {
public:
    MetaCacheBasedMetaAdaptorFactory(cache::CachePtr metaCache) : metaCache_(std::move(metaCache)) {
    }

private:
    db::MetaMasterProviderPtr getMetaMasterProvider(const db::AdaptorConfig&,
                                                    const db::MetaPoolPtr&,
                                                    const Scribe& scribe) const override {
        return std::make_shared<db::MetaCacheBasedMetaMasterProvider>(metaCache_, scribe);
    }

    cache::CachePtr metaCache_;
};

db::MetaAdaptorFactoryPtr getMetaAdaptorFactory(ConfigPtr config, cache::CachePtr metaCache) {
    if (config->metaPoller.enabled) {
        LOGDOG_(
            log::GetLogger(log::sharpeiLogKey), notice, log::message = "MetaCacheBasedMetaAdaptorFactory will be used");
        return std::make_shared<MetaCacheBasedMetaAdaptorFactory>(metaCache);
    } else {
        LOGDOG_(
            log::GetLogger(log::sharpeiLogKey), notice, log::message = "ReplMonBasedMetaAdaptorFactory will be used");
        return std::make_shared<ReplMonBasedMetaAdaptorFactory>();
    }
}

}  // namespace

namespace sharpei {

void Module::init(const yplatform::ptree& data) {
    using sharpei::cache::Cache;

    config_ = makeConfig(data);
    shardsCache_ = std::make_shared<Cache>(config_->cache.historyCapacity, config_->cache.errorsLimit);

    metaCache_ = std::make_shared<Cache>(config_->cache.historyCapacity, config_->cache.errorsLimit);
    auto metaEndpointProvider = getEndpointProvider(config_, metaCache_);

    if (config_->meta.endpointProvider.useMetaCacheBasedProviderInMetaPool) {
        metaPool_ = db::makeMetaPool(config_->meta.db.pool, metaEndpointProvider);
    } else {
        metaPool_ = db::makeMetaPool(config_->meta.db.pool, db::makeSingleEndpointProvider(config_->meta));
    }

    metaAdaptorFactory_ = ::getMetaAdaptorFactory(config_, metaCache_);

    shardWorkers_ = std::make_unique<ThreadPool>(config_->shard.workers);
    shardPool_ = db::makeConnectionPool(config_->shard.pool, shardWorkers_->io());
    peersPool_ = db::makeConnectionPool(config_->peers.pool, peersIos_);
    metaPoller_ = std::make_unique<poller::MetaPoller>(config_->metaPoller.base, config_->meta, metaCache_);

    shardsPoller_ = std::make_unique<poller::ShardsPoller>(config_->shardsPoller,
                                                           config_->meta,
                                                           config_->shard,
                                                           std::move(metaEndpointProvider),
                                                           shardPool_,
                                                           shardsCache_,
                                                           metaAdaptorFactory_);

    if (!pa::async_profiler::is_init()) {
        pa::async_profiler::init(1000000, 16000, config_->pa.log);
    }

    LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message="module initialized");
}

void Module::start() {
    shardWorkers_->start();
    peersWorker_ = std::make_unique<IoThreadWorker>("peers_worker", peersIos_);
    if (config_->metaPoller.enabled) {
        metaPoller_->start();
    }
    shardsPoller_->start();
    LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message="module started");
}

void Module::stop() {
    shardsPoller_->stop();
    peersWorker_->stop();
    shardWorkers_->stop();
    if (config_->metaPoller.enabled) {
        metaPoller_->stop();
    }
    LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message="module stopped");
}

void Module::fini() {
    shardsPoller_.reset();
    peersIos_.stop();
    peersWorker_.reset();
    shardWorkers_.reset();
    peersPool_.reset();
    shardPool_.reset();
    metaPool_.reset();
    shardsCache_.reset();
    metaCache_.reset();
    config_.reset();
    LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message="module finished");
}

yplatform::ptree Module::get_stats() const {
    yplatform::ptree result;
    stats::addMetaPool(metaPool_, result);
    stats::addShardPool(shardPool_, result);
    stats::addPeersPool(peersPool_, result);
    return result;
}

ConfigPtr Module::getConfig() const {
    return config_;
}

cache::CachePtr Module::getShardsCache() const {
    return shardsCache_;
}

cache::CachePtr Module::getMetaCache() const {
    return metaCache_;
}

db::MetaPoolPtr Module::getMetaPool() const {
    return metaPool_;
}

db::ShardPoolPtr Module::getShardPool() const {
    return shardPool_;
}

db::PeersPoolPtr Module::getPeersPool() const {
    return peersPool_;
}

db::MetaAdaptorFactoryPtr Module::getMetaAdaptorFactory() const {
    return metaAdaptorFactory_;
}

} // namespace sharpei
