#include <internal/poller/poller.h>
#include <internal/helpers.h>
#include <internal/async_operation.h>
#include <internal/get_io_context.h>

#include <spdlog/details/format.h>

namespace {

sharpei::Error errorForPollerType(sharpei::poller::Poller::Type type) {
    using namespace sharpei;
    switch (type) {
        case poller::Poller::Type::cluster:
            return Error::clusterPollingError;
        case poller::Poller::Type::meta:
            return Error::metaPollingError;
        case poller::Poller::Type::shards:
            return Error::shardsPollingError;
    }
}

sharpei::Error makeResetError(sharpei::poller::Poller::Type type) {
    using namespace sharpei;
    switch (type) {
        case poller::Poller::Type::cluster:
            return Error::clusterPollingResetCacheError;
        case poller::Poller::Type::meta:
            return Error::metaPollingResetCacheError;
        case poller::Poller::Type::shards:
            return Error::shardsPollingResetCacheError;
    }
}

std::string toString(sharpei::poller::Poller::Type type) {
    using namespace sharpei;
    switch (type) {
        case poller::Poller::Type::cluster:
            return "cluster";
        case poller::Poller::Type::meta:
            return "meta";
        case poller::Poller::Type::shards:
            return "shards";
    }
}

}

namespace sharpei::poller {

Poller::Poller(Poller::Type type, std::string uniqId, const PollerConfig& config, ShardsProviderPtr shardsProvider,
           db::ShardAdaptorPtr shardAdaptor, cache::CachePtr cache)
        : type(type),
          uniqId(std::move(uniqId)),
          config(config),
          shardsProvider(shardsProvider),
          shardAdaptor(shardAdaptor),
          cache(cache),
          shouldStop(false) {
}

void Poller::start(boost::asio::io_context& io) {
    PromisePtr readinessPromise{new PromisePtr::element_type{}};
    auto readinessFuture{readinessPromise->get_future()};

    shouldStop = false;

    boost::asio::spawn(io, [self = shared_from_this(), readinessPromise = std::move(readinessPromise)] (boost::asio::yield_context yield) {
        boost::uuids::random_generator generator;
        const auto context = makeTaskContext(self->uniqId, generateRequestId(generator), yield);

        LOGDOG_(context->scribe().logger, notice, log::message="poller started");

        boost::asio::steady_timer timer(getIoContext(yield));

        auto doPoll = [&](PromisePtr promise) {
            bool ok = false;
            try {
                const auto pollContext = makeTaskContext(context->uniq_id(), generateRequestId(generator), yield);

                boost::system::error_code ec;
                timer.expires_from_now(self->config.interval);
                timer.async_wait(yield[ec]);

                if (ec) {
                    LOGDOG_(pollContext->scribe().logger, error, log::message="shards poller timer error",
                            log::error_code=errorForPollerType(self->type), log::reason=ec);
                }

                self->poll(pollContext, std::move(promise))
                    .bind([&]() { ok = true; } )
                    .catch_error([&] (const auto& error) {
                        LOGDOG_(pollContext->scribe().logger, error, log::message="error when polling shards",
                                log::error_code=errorForPollerType(self->type), log::reason=error);
                    });
            } catch (const std::exception& e) {
                LOGDOG_(context->scribe().logger, error, log::message="exception when polling shards",
                        log::error_code=errorForPollerType(self->type), log::reason=std::cref(e));
            } catch (const boost::coroutines::detail::forced_unwind&) {
                throw;
            } catch (...) {
                LOGDOG_(context->scribe().logger, error, log::reason="unknown exception when polling shards",
                        log::error_code=errorForPollerType(self->type));
            }
            return ok;
        };

        // We'll spin around until the first successful run of |resetCache(...)|, otherwise there will
        // be no guarantee that all callbacks will be called and future eventually become ready.
        while (!self->shouldStop) {
            if (doPoll(readinessPromise)) {
                break;
            }
        }

        while (!self->shouldStop) {
            doPoll(nullptr);
        }

        LOGDOG_(context->scribe().logger, notice, log::message="poller stopped");
    }, boost::coroutines::attributes(config.coroutineStackSize));

    LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message=fmt::format("awaiting {} cache...", toString(type)));
    readinessFuture.wait();
    LOGDOG_(log::GetLogger(log::sharpeiLogKey), notice, log::message=fmt::format("{} cache is ready", toString(type)));
}

void Poller::stop() {
    shouldStop = true;
}

expected<void> Poller::poll(const TaskContextPtr& context, PromisePtr promise) {
    const auto resetCache = [&] (const auto& shards) { return this->resetCache(context, shards, std::move(promise)); };

    return shardsProvider->getAllShards(context)
        .bind(resetCache);
}

expected<void> Poller::resetCache(const TaskContextPtr& context, const std::vector<ShardWithoutRoles>& shards, PromisePtr promise) {
    auto countdown = std::make_shared<Countdown<>>(shards.size(), [promise = std::move(promise)]() {
        if (promise) {
            promise->set_value();
        }
    });

    for (const auto& shard : shards) {
        const auto onValue = [context, id = shard.id, countdown]() {
            countdown->decrease();
            LOGDOG_(context->scribe().logger, debug, log::message="reset cache for shard", log::shard_id=id);
        };

        const auto onError = [self = shared_from_this(), context, id = shard.id, countdown](const auto& error) {
            countdown->decrease();
            if (error == sharpei::make_error_code(Error::resetError)) {
                LOGDOG_(context->scribe().logger, warning, log::message = "error when polling shard", log::shard_id = id,
                        log::error_code = makeResetError(self->type));
            } else {
                LOGDOG_(context->scribe().logger, error, log::message = "error when polling shard", log::shard_id = id,
                        log::error_code = errorForPollerType(self->type), log::reason = error);
            }
        };

        shardAdaptor->resetHostCache(shard, onValue, onError);
    }

    return {};
}

} // namespace sharpei::poller
