#include <internal/services/blackbox/blackbox_impl.h>
#include <internal/services/utils.h>
#include <internal/errors.h>

#include <yamail/data/deserialization/yajl.h>

BOOST_FUSION_ADAPT_STRUCT(sharpei::services::blackbox::HostedDomain,
    domid
)

BOOST_FUSION_DEFINE_STRUCT((sharpei)(services)(blackbox), HostedDomainResponse,
    (std::vector<sharpei::services::blackbox::HostedDomain>, hosted_domains)
)

namespace sharpei::services::blackbox {
namespace {

template <class T>
sharpei::expected<T> fromJson(const std::string& value, const TaskContextPtr& context) {
    using namespace sharpei;
    using yamail::data::deserialization::fromJson;
    try {
        return fromJson<T>(value);
    } catch (const std::exception& e) {
        LOGDOG_(context->scribe().logger, error, log::message="failed to parse blackbox response", log::exception=e, log::body=value);
        return make_unexpected(ExplainedError(Error::blackBoxParseError));
    }
}


} // namespace

BlackboxImpl::BlackboxImpl(BlackboxConfig config, std::shared_ptr<ymod_httpclient::cluster_call> httpClient)
    : config(std::move(config)),
      httpClient(std::move(httpClient)) {
}

expected<std::vector<HostedDomain>> BlackboxImpl::getHostedDomains(DomainId domainId,
        const TaskContextPtr& context) const {
    return getHostedDomainsImpl(domainId, context);
}

expected<std::vector<HostedDomain>> BlackboxImpl::getHostedDomains(const std::string& domain,
        const TaskContextPtr& context) const {
    return getHostedDomainsImpl(domain, context);
}

expected<std::vector<HostedDomain>> BlackboxImpl::parse(const std::string& body,
        const TaskContextPtr& context) const {
    return fromJson<HostedDomainResponse>(body, context)
        .bind([] (auto parsed) { return parsed.hosted_domains; });
}

} // namespace sharpei::services::blackbox
