#include <internal/services/iam/iam_client.h>
#include <internal/errors.h>

#include <yamail/data/serialization/yajl.h>
#include <yamail/data/deserialization/yajl.h>

#include <butil/http/headers.h>

BOOST_FUSION_DEFINE_STRUCT((sharpei)(services)(iam)(detail), PostIamTokensRequest,
    (std::string_view, jwt)
)

BOOST_FUSION_DEFINE_STRUCT((sharpei)(services)(iam)(detail), PostIamTokensResponse,
    (std::string, iamToken)
)

namespace sharpei::services::iam {
namespace {

template <class T>
expected<T> fromJson(const std::string& value, const TaskContextPtr& context) {
    using yamail::data::deserialization::fromJson;
    try {
        return fromJson<T>(value);
    } catch (const std::exception& e) {
        LOGDOG_(context->scribe().logger, error, log::message="failed to parse iam response", log::exception=e, log::body=value);
        return make_unexpected(ExplainedError(Error::iamParseError));
    }
}

} // namespace

expected<IamToken> IamClient::getIamToken(const TaskContextPtr& context) const {
    return jwtGenerator->generate(context)
        .bind([&] (auto token) { return getIamToken(context, token); });
}

expected<IamToken> IamClient::getIamToken(const TaskContextPtr& context, std::string_view jwt) const {
    using yamail::data::serialization::toJson;

    http::headers headers;
    headers.add("Content-Type", "application/json");
    headers.add("X-Request-Id", context->requestId());

    const auto request = yhttp::request::POST(
        "/v1/tokens",
        headers.format(),
        toJson(detail::PostIamTokensRequest {jwt})
    );

    boost::system::error_code ec;

    const auto response = httpClient->async_run(
        context,
        request,
        config.http.options,
        context->yield()[ec]
    );

    if (ec) {
        LOGDOG_(context->scribe().logger, error, log::url=request.url, log::error_code=ec);
        return make_unexpected(ExplainedError(Error::iamHttpError));
    }

    if (response.status != 200) {
        LOGDOG_(context->scribe().logger, error, log::url=request.url, log::body=response.body);
        return make_unexpected(ExplainedError(Error::iamHttpError));
    }

    return fromJson<detail::PostIamTokensResponse>(response.body, context)
        .bind([] (auto&& value) -> expected<IamToken> { return {IamToken {std::move(value.iamToken)}}; });
}

} // namespace sharpei::services::iam
