#include <internal/services/iam/iam_poller.h>
#include <internal/get_io_context.h>

namespace sharpei::services::iam {

void IamPoller::start(boost::asio::io_context& io) {
    shouldStop = false;
    boost::asio::spawn(io, [self = shared_from_this()] (boost::asio::yield_context yield) {
        boost::uuids::random_generator generator;
        const auto context = makeTaskContext(self->uniqId, generateRequestId(generator), yield);

        LOGDOG_(context->scribe().logger, notice, log::message="iam poller is started");

        boost::asio::steady_timer timer(getIoContext(yield));

        while (!self->shouldStop) {
            try {
                const auto pollContext = makeTaskContext(self->uniqId, generateRequestId(generator), yield);

                const auto result = self->client->getIamToken(pollContext)
                    .bind([&] (auto&& token) {
                        self->iamTokenCache->emplace(std::move(token));
                        self->intervalGenerator.reset();
                    })
                    .catch_error([&] (auto&& error) {
                        self->intervalGenerator.decrease();
                        LOGDOG_(context->scribe().logger, warning, log::message="get iam token error", log::error_code=error);
                    });

                boost::system::error_code ec;

                const auto interval = self->intervalGenerator.generate();

                timer.expires_from_now(interval);
                timer.async_wait(yield[ec]);

                if (ec) {
                    LOGDOG_(pollContext->scribe().logger, warning, log::message="iam poller timer error", log::error_code=ec);
                }
            } catch (const std::exception& e) {
                LOGDOG_(context->scribe().logger, error, log::message="iam poller exception", log::exception=e);
            } catch (const boost::coroutines::detail::forced_unwind&) {
                throw;
            } catch (...) {
                LOGDOG_(context->scribe().logger, error, log::message="iam poller unknown exception");
            }
        }

        LOGDOG_(context->scribe().logger, notice, log::message="iam poller is finished");
    }, boost::coroutines::attributes(config.coroutineStackSize));
}

void IamPoller::stop() {
    shouldStop = true;
}

} // namespace sharpei::services::iam
