#include <internal/services/yc/yc_client.h>
#include <internal/services/yc/reflection.h>
#include <internal/errors.h>

#include <yamail/data/deserialization/yajl.h>

#include <butil/http/headers.h>

#include <boost/range/algorithm/copy.hpp>

BOOST_FUSION_DEFINE_STRUCT((sharpei)(services)(yc)(detail), GetHostsResponse,
    (std::vector<sharpei::services::yc::Host>, hosts)
)

namespace sharpei::services::yc {
namespace {

template <class T>
expected<T> fromJson(const std::string& value, const TaskContextPtr& context) {
    using yamail::data::deserialization::fromJson;
    try {
        return fromJson<T>(value);
    } catch (const std::exception& e) {
        LOGDOG_(context->scribe().logger, error, log::message="failed to parse yc response", log::exception=e, log::body=value);
        return make_unexpected(ExplainedError(Error::ycParseError));
    }
}

} // namespace

expected<std::vector<Host>> YcClient::getHosts(const IamToken& iamToken, const TaskContextPtr& context) const {
    using namespace std::string_literals;

    http::headers headers;
    auto authorization = "Bearer "s;
    boost::copy(static_cast<const std::string_view&>(iamToken), std::back_inserter(authorization));
    headers.add("Authorization", authorization);
    headers.add("X-Request-Id", context->requestId());

    const auto request = yhttp::request::GET(
        "/managed-postgresql/v1/clusters/"s + config.clusterId + "/hosts"s,
        headers.format()
    );

    boost::system::error_code ec;

    const auto response = httpClient->async_run(
        context,
        request,
        config.http.options,
        context->yield()[ec]
    );

    if (ec) {
        LOGDOG_(context->scribe().logger, error, log::url=request.url, log::error_code=ec);
        return make_unexpected(ExplainedError(Error::ycHttpError));
    }

    if (response.status != 200) {
        LOGDOG_(context->scribe().logger, error, log::url=request.url, log::body=response.body);
        return make_unexpected(ExplainedError(Error::ycHttpError));
    }

    return fromJson<detail::GetHostsResponse>(response.body, context)
        .bind([] (auto&& value) -> expected<std::vector<Host>> { return {std::move(value.hosts)}; });
}

} // namespace sharpei::services::yc
