#include <internal/services/yc/yc_poller.h>
#include <internal/services/yc/reflection.h>
#include <internal/get_io_context.h>

#include <yamail/data/serialization/json_writer.h>

#include <boost/filesystem.hpp>

#include <fstream>

namespace sharpei::services::yc {

void YcPoller::start(boost::asio::io_context& io) {
    shouldStop = false;
    boost::asio::spawn(io, [self = shared_from_this()] (boost::asio::yield_context yield) {
        boost::uuids::random_generator generator;
        const auto context = makeTaskContext(self->uniqId, generateRequestId(generator), yield);

        LOGDOG_(context->scribe().logger, notice, log::message="yc poller is started");

        boost::asio::steady_timer timer(getIoContext(yield));

        while (!self->shouldStop) {
            try {
                const auto pollContext = makeTaskContext(self->uniqId, generateRequestId(generator), yield);

                if (const auto cachedIamToken = *self->iamTokenCache) {
                    const IamToken iamToken {static_cast<const std::string&>(*cachedIamToken)};

                    self->client->getHosts(iamToken, pollContext)
                        .bind([&] (auto&& hosts) {
                            self->hostsCache->emplace(std::move(hosts));
                            self->asyncWriteFsHostsCache(pollContext);
                        })
                        .catch_error([&] (const auto& error) {
                            LOGDOG_(pollContext->scribe().logger, warning, log::message="get hosts from yc error", log::error_code=error);
                        });
                } else {
                    LOGDOG_(pollContext->scribe().logger, warning, log::message="yc poller error: no iam token, can't get hosts");
                }

                boost::system::error_code ec;

                const auto interval = self->config.interval;

                timer.expires_from_now(interval);
                timer.async_wait(yield[ec]);

                if (ec) {
                    LOGDOG_(pollContext->scribe().logger, warning, log::message="yc poller timer error", log::error_code=ec);
                }
            } catch (const std::exception& e) {
                LOGDOG_(context->scribe().logger, error, log::message="yc poller exception", log::exception=e);
            } catch (const boost::coroutines::detail::forced_unwind&) {
                throw;
            } catch (...) {
                LOGDOG_(context->scribe().logger, error, log::message="yc poller unknown exception");
            }
        }

        LOGDOG_(context->scribe().logger, notice, log::message="yc poller is finished");
    }, boost::coroutines::attributes(config.coroutineStackSize));
}

void YcPoller::stop() {
    shouldStop = true;
    if (writeFsHostsCacheResult) {
        writeFsHostsCacheResult->get();
    }
}

void YcPoller::asyncWriteFsHostsCache(const TaskContextPtr& context) {
    if (shouldStop) {
        return;
    }

    if (writeFsHostsCacheResult && writeFsHostsCacheResult->wait_for(std::chrono::seconds(0)) != std::future_status::ready) {
        return;
    }

    writeFsHostsCacheResult = std::async(std::launch::async, [self = shared_from_this(), hosts = hostsCache->value(), context] {
        using yamail::data::serialization::writeJson;
        using namespace std::literals;

        if (self->shouldStop) {
            return;
        }

        const auto tempPath = self->config.fsHostsCachePath + ".temp";

        {
            std::ofstream file(tempPath);

            if (!file.is_open()) {
                LOGDOG_(context->scribe().logger, warning, log::message="can't open file to write hosts cache: \""s + tempPath + "\"");
                return;
            }

            writeJson(file, hosts);

            if (file.bad() || file.fail()) {
                LOGDOG_(context->scribe().logger, warning, log::message="write hosts cache to file error: \""s + tempPath + "\"");
                return;
            }
        }

        boost::system::error_code ec;
        boost::filesystem::rename(tempPath, self->config.fsHostsCachePath, ec);

        if (ec) {
            LOGDOG_(context->scribe().logger, warning, log::message="rename from \""s + tempPath
                    + "\" to \"" + self->config.fsHostsCachePath + "\" hosts cache file error",
                    log::error_code=ec);
            return;
        }

        LOGDOG_(context->scribe().logger, notice, log::message="hosts cache is written to file: \""s + self->config.fsHostsCachePath + "\"");
    });
}

} // namespace sharpei::services::yc
