#pragma once

#include <mail/unistat/cpp/include/meters/common.h>
#include <mail/unistat/cpp/include/meters/request_matcher.h>
#include <mail/unistat/cpp/include/signal_record.h>

#include <util/generic/yexception.h>
#include <util/system/yassert.h>

#include <boost/lexical_cast.hpp>

#include <chrono>
#include <cstdlib>
#include <string>
#include <string_view>

namespace unistat {

class ShardHostStateMatcher {
public:
    ShardHostStateMatcher(std::string_view shard, std::string_view host, std::string_view state)
    : my(shard, host, state) {
    }

    bool match(std::string_view shard) const noexcept {
        return std::get<0>(my) == shard;
    }

    bool match(std::string_view shard, std::string_view host, std::string_view state) const noexcept {
        return my == std::make_tuple(shard, host, state);
    }

private:
    std::tuple<std::string, std::string, std::string> my;
};

struct YmodWebserverAccessLogAdaptor {
    static std::optional<std::chrono::milliseconds> getExecTime(const std::map<std::string, std::string>& record);

    static std::optional<std::string_view> getRequest(const std::map<std::string, std::string>& record);

    static std::optional<long> getStatus(const std::map<std::string, std::string>& record);
};

struct HttpClientLogAdaptor {
    static std::optional<std::string_view> getRequest(const std::map<std::string, std::string>& record);

    static std::optional<long> getStatus(const std::map<std::string, std::string>& record);
};

struct PaLogAdaptor {
    static std::optional<std::string_view> getRequest(const PaRecord& record);

    static std::optional<std::chrono::milliseconds> getExecTime(const PaRecord& record);

    static std::optional<std::string_view> getEndpoint(const PaRecord& record);
};

struct StatusLogAdaptor {
    using Unixtime = unsigned long long;

    static std::optional<std::string_view> getHost(const std::map<std::string, std::string>& record);

    static std::optional<std::string_view> getState(const std::map<std::string, std::string>& record);

    static std::optional<std::string_view> getShardName(const std::map<std::string, std::string>& record);

    static std::optional<Unixtime> getUnixtime(const std::map<std::string, std::string>& record);
};

/*
 * Returns 1 iff there was an entry in logs with the given (shard, host, aliveness) within ttl seconds from "now".
 * "now" = value of the "unixtime" field of the last line read. This way alert will not be fired if we stop the service.
 * For alive hosts we aggregate using minimum (annn) and maximum (axxx) for dead.
 */
template <typename Matcher, typename LogAdaptor>
struct StatusRecordsCounter {
    StatusRecordsCounter(std::string_view namePrefix, std::string_view shard, std::string_view host, bool alive)
    : namePrefix(normalizeName(buildSignalName(namePrefix, stateToString(alive), host))),
      matcher(shard, host, stateToString(alive)), alive(alive) {
    }

    template <typename T>
    void update(const T& arg) {
        if (auto shardName = LogAdaptor::getShardName(arg); shardName) {
            if (auto host = LogAdaptor::getHost(arg); host) {
                if (auto state = LogAdaptor::getState(arg); state) {
                    update(LogAdaptor::getUnixtime(arg).value(), *shardName, *host, *state);
                }
            }
        }
    }

    NamedValue<unsigned> get() const {
        Y_VERIFY(!lastOccurence.has_value() || (now.has_value() && *now >= *lastOccurence));
        const bool result = lastOccurence.has_value() && ((*now - *lastOccurence) <= ttl);
        const auto suffix = alive ? "annn" : "axxx";
        return {withSigoptSuffix(namePrefix, suffix), result};
    }

private:
    using Unixtime = StatusLogAdaptor::Unixtime;

    static std::string stateToString(bool alive) {
        return alive ? "alive" : "dead";
    }

    std::string buildSignalName(std::string_view namePrefix, std::string_view state, std::string_view host) {
        const auto name = std::string(namePrefix) + "_" + std::string(state);
        return ::unistat::buildSignalName(name, host);
    }

    void update(Unixtime unixtime, std::string_view shardName, std::string_view host, std::string_view state) {
        if (matcher.match(shardName)) {
            // The following condition should hold only within each thread separately.
            // Since each shard is processed by exactly one thread, it is sufficient to check the |shardName|'s value.
            Y_VERIFY(unixtime >= now);
            now = unixtime;
            if (matcher.match(shardName, host, state)) {
                lastOccurence = unixtime;
            }
        }
    }

    static constexpr auto ttl = 3;
    const std::string namePrefix;
    const Matcher matcher;
    const bool alive;

    std::optional<Unixtime> now;
    std::optional<Unixtime> lastOccurence;
};

template <typename Matcher, typename LogAdaptor, long... ShouldAlwaysBePresent>
struct ByStatusCodeRequestCounter {
    ByStatusCodeRequestCounter(std::string_view handler, std::string_view namePrefix)
    : namePrefix(normalizeName(buildSignalName(namePrefix, handler))), matcher(handler) {
        for (const auto shouldAlwaysBePresent : {ShouldAlwaysBePresent...}) {
            counter[shouldAlwaysBePresent] = 0;
        }
    }

    template <typename T>
    void update(const T& arg) {
        if (auto request = LogAdaptor::getRequest(arg); request) {
            if (auto status = LogAdaptor::getStatus(arg); status) {
                update(*request, *status);
            }
        }
    }

    auto get() const {
        std::vector<NamedValue<size_t>> res;
        res.reserve(counter.size());
        for (auto [status, count] : counter) {
            res.emplace_back(withSigoptSuffix(namePrefix + std::to_string(status), "summ"), count);
        }
        return res;
    }

private:
    void update(std::string_view request, long status) {
        Y_ENSURE_EX(!status || (200 <= status && status < 600), TBadArgumentException{} << status);
        if (matcher.match(request)) {
            ++counter[status];
        }
    }

    const std::string namePrefix;
    std::unordered_map<long, uint64_t> counter;
    Matcher matcher;
};

template <typename Matcher, typename LogAdaptor>
struct Histogram {
    using BorderValue = std::chrono::milliseconds::rep;
    using Impl = Hist<BorderValue>;

    Histogram(std::vector<BorderValue> borders, std::string_view handler, std::string_view namePrefix)
    : impl(std::move(borders), normalizeName(buildSignalName(namePrefix, handler))), matcher(handler) {
    }

    template <typename T>
    void update(const T& arg) {
        if (auto request = LogAdaptor::getRequest(arg); request) {
            if (auto status = LogAdaptor::getExecTime(arg); status) {
                update(*request, *status);
            }
        }
    }

    auto get() const {
        return impl.get();
    }

private:
    void update(std::string_view request, std::chrono::milliseconds time) {
        if (matcher.match(request)) {
            impl.update(time.count());
        }
    }

    Impl impl;
    Matcher matcher;
};

template <typename HistImpl>
struct HistogramForEndpoint {
    using BorderValue = typename HistImpl::BorderValue;

    HistogramForEndpoint(std::vector<BorderValue> borders, std::string_view handler, std::string_view namePrefix,
                         std::string_view endpoint)
    : impl(std::move(borders), handler, buildSignalName(namePrefix, endpoint)), endpoint(truncateByHostLength(endpoint)) {
    }

    template <typename T>
    void update(const T& arg) {
        if (auto ep = PaLogAdaptor::getEndpoint(arg); ep) {
            if (endpoint == *ep) {
                impl.update(arg);
            }
        }
    }

    auto get() const {
        return impl.get();
    }

private:
    static std::string truncateByHostLength(std::string_view endpoint) {
        static constexpr size_t maxHostLength = std::size(decltype(PaRecord::host){});
        return std::string(endpoint).substr(0, maxHostLength);
    }

    HistImpl impl;
    const std::string endpoint;
};

using AccessLogRequestCounter =
    ByStatusCodeRequestCounter<PathMatcher, YmodWebserverAccessLogAdaptor, 200, 401, 404, 500, 502, 503, 504>;

using AccessLogHistogram = Histogram<PathMatcher, YmodWebserverAccessLogAdaptor>;

using AccessLogHistogramWeak = Histogram<SubstringMatcher, YmodWebserverAccessLogAdaptor>;

using HttpClientLogRequestCounter =
    ByStatusCodeRequestCounter<SubstringMatcher, HttpClientLogAdaptor, 0, 200, 401, 404, 500, 502, 503, 504>;

using PaLogHistogram = Histogram<SubstringMatcher, PaLogAdaptor>;

using PaLogHistogramForEndpoint = HistogramForEndpoint<PaLogHistogram>;

using StatusCounter = StatusRecordsCounter<ShardHostStateMatcher, StatusLogAdaptor>;

}  // namespace unistat
