#ifndef SHARPEI_CLIENT_ERRORS_H
#define SHARPEI_CLIENT_ERRORS_H

#include <stdexcept>
#include <mail_errors/error_code.h>

namespace sharpei {
namespace client {

struct Exception: public std::runtime_error {
    Exception(const std::string& msg)
        : std::runtime_error(msg)
    {}
};

enum class Errors {
    RetriesLimit = 1,
    HttpCode,
    Exception,
    UidNotFound,
    TimerError,
    SharpeiError,
    EmptyShard,
    RegistrationInProgress,
    ShardNotFound,
};

} // namespace client
} // namespace sharpei

namespace boost {
namespace system {

template <>
struct is_error_code_enum<sharpei::client::Errors> {
    static const bool value = true;
};

} // namespace system
} // namespace boost

namespace sharpei {
namespace client {

class ErrorCategory: public boost::system::error_category {
public:
    const char* name() const noexcept {
        return "sharpei_client";
    }

    std::string message(int error) const {
        const std::string ret = "sharpei_client error: ";
        switch(Errors(error)) {
            case Errors::RetriesLimit: return ret + "retries limit";
            case Errors::HttpCode: return ret + "bad http code";
            case Errors::Exception: return ret + "exception was caught";
            case Errors::UidNotFound: return ret + "uid not found";
            case Errors::TimerError: return ret + "timer problem";
            case Errors::SharpeiError: return ret + "sharpei error";
            case Errors::EmptyShard: return ret + "empty shard in response";
            case Errors::RegistrationInProgress: return ret + "registration in progress";
            case Errors::ShardNotFound: return ret + "shard not found";
        }

        return ret + "unknown error";
    }
};

inline const ErrorCategory& getErrorCategory() {
    static ErrorCategory cat;
    return cat;
}

using ErrorCode = mail_errors::error_code;

inline auto make_error_code(Errors e) {
    return ErrorCode::base_type(static_cast<int>(e), getErrorCategory());
}

inline ErrorCode makeErrorCode(Errors error, const std::string& msg = "") {
    return ErrorCode(make_error_code(error), msg);
}

} // namespace
} // namespace

#endif // SHARPEI_CLIENT_ERRORS_H
