#pragma once

#include <butil/http/arguments.h>
#include <butil/http/headers.h>

#include <sharpei_client/http.h>
#include <ymod_httpclient/cluster_client.h>

namespace sharpei::internal {

using namespace ::sharpei::client::http;

class DefaultHttpClient : public HttpClient {
public:
    using YHttpClient = yhttp::cluster_client;
    using YHttpClientPtr = std::shared_ptr<YHttpClient>;

    explicit DefaultHttpClient(const ClientSettings &settings) {
        if (settings.clusterClientModuleName.empty()) {
            throw std::invalid_argument("clusterClientModuleName is empty");
        }
        httpClient_ = yplatform::find<YHttpClient, std::shared_ptr>(settings.clusterClientModuleName);
    }

    void aget(const Address &addr,
              Timeout timeout,
              const std::string &method,
              const Arguments &args,
              const Headers &headers,
              ResponseHandler handler,
              bool keepAlive,
              const std::string &requestId) const override {
        const auto creator = [](std::string &&url, std::string &&headers) {
            return yhttp::request::GET(std::move(url), std::move(headers));
        };
        asyncRequest(addr, timeout, method, args, headers, std::move(handler), keepAlive, requestId, creator);
    }

    void apost(const Address &addr,
               Timeout timeout,
               const std::string &method,
               const Arguments &args,
               const Headers &headers,
               const std::string &data,
               ResponseHandler handler,
               bool keepAlive,
               const std::string &requestId) const override {
        const auto creator = [&data](std::string &&url, std::string &&headers) {
            return yhttp::request::POST(std::move(url), std::move(headers), std::string(data));
        };
        asyncRequest(addr, timeout, method, args, headers, std::move(handler), keepAlive, requestId, creator);
    }

private:
    template <typename RequestCreator>
    void asyncRequest(const Address & /*addr*/,
                      Timeout /*timeout*/,
                      const std::string &method,
                      const Arguments &args,
                      const Headers &headers,
                      ResponseHandler handler,
                      bool /*keepAlive*/,
                      const std::string &requestId,
                      RequestCreator &requestCreate) const {
        std::string url = fmt::format("{}?{}", method, HttpArguments{args}.format());
        std::string headersStr = http::headers{headers}.format();

        httpClient_->async_run(boost::make_shared<yplatform::task_context>(requestId),
                               requestCreate(std::move(url), std::move(headersStr)),
                               [handler = std::move(handler)](boost::system::error_code ec, yhttp::response response) {
                                   handler(ec,
                                           ec ? Response{} :
                                                Response{.code = static_cast<unsigned>(response.status),
                                                         .body = std::move(response.body)});
                               });
    }

    YHttpClientPtr httpClient_;
};

}
