#pragma once

#include <unordered_map>
#include <functional>
#include <chrono>
#include <boost/optional.hpp>
#include <sharpei_client/http.h>
#include <sharpei_client/errors.h>
#include <io_result/io_result.h>

namespace sharpei {
namespace client {

using UserId = std::string;

struct Settings {
    http::Address sharpeiAddress;
    http::Timeout timeout = http::Timeout{0};
    unsigned retries = 0;
    bool keepAlive = false;

    std::optional<http::ClientSettings> httpClient;

    using HttpExceptionHandler = std::function<void(const std::exception&)>;
    HttpExceptionHandler onHttpException = [] (const std::exception&) {};
    using HttpErrorHandler = std::function<void(const ErrorCode& error, const std::string&)>;
    HttpErrorHandler onHttpError = [] (const ErrorCode&, const std::string&) {};
};

struct RequestInfo {
    std::string requestId;
    std::string connectionId;
    std::string clientType;
    std::string userIp;
    std::string uniqId;
};

enum class Mode {
    WriteOnly,
    WriteRead,
    ReadWrite,
    ReadOnly,
    All
};

struct ResolveParams {
    using Uid = std::string;
    ResolveParams (const Uid& u, Mode mode = Mode::WriteOnly, bool force = true)
        : uid(u), mode(mode), force(force) { }

    Uid uid;
    Mode mode;
    bool force;
};

struct Shard {
    using Id = std::string;
    struct Database {
        struct Address {
            std::string host;
            unsigned port;
            std::string dbname;
            std::string dataCenter;

            std::string toString() const;
        };

        struct State {
            std::int32_t lag;
        };

        Address address;
        std::string role;
        std::string status;
        State state;
    };

    Id id;
    std::string name;
    std::vector<Database> databases;
};

struct ShardWithUid {
    std::string uid;
    Shard shard;
};

using OptShard = boost::optional<Shard>;
using MapShard = std::unordered_map<Shard::Id, Shard>;

namespace io = ::io_result;

class SharpeiClient {
public:
    using AsyncHandler = io::Hook<Shard>;
    using AsyncMapHandler = io::Hook<MapShard>;

    virtual ~SharpeiClient() {}

    template <typename Handler = io::sync_context>
    auto getConnInfo(const ResolveParams& p, Handler h = io::use_sync) const {
        io_result::detail::init_async_result<Handler, AsyncHandler> init(h);
        asyncGetConnInfo(p, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getDeletedConnInfo(const ResolveParams& p, Handler h = io::use_sync) const {
        io_result::detail::init_async_result<Handler, AsyncHandler> init(h);
        asyncGetDeletedConnInfo(p, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getOrgConnInfo(const ResolveParams& p, Handler h = io::use_sync) const {
        io_result::detail::init_async_result<Handler, AsyncHandler> init(h);
        asyncGetOrgConnInfo(p, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto stat(Handler h = io::use_sync) const {
        io_result::detail::init_async_result<Handler, AsyncMapHandler> init(h);
        asyncStat(init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto statById(const Shard::Id& id, Handler h = io::use_sync) const {
        io_result::detail::init_async_result<Handler, AsyncHandler> init(h);
        asyncStatById(id, init.handler);
        return init.result.get();
    }

    virtual void asyncGetConnInfo(const ResolveParams& params, AsyncHandler handler) const = 0;
    virtual void asyncGetDeletedConnInfo(const ResolveParams& params, AsyncHandler handler) const = 0;
    virtual void asyncGetOrgConnInfo(const ResolveParams& params, AsyncHandler handler) const = 0;
    virtual void asyncStat(AsyncMapHandler handler) const = 0;
    virtual void asyncStatById(const Shard::Id& id, AsyncHandler handler) const = 0;
};

using SharpeiClientPtr = std::shared_ptr<const SharpeiClient>;

SharpeiClientPtr createSharpeiClient(Settings settings, const RequestInfo& requestInfo);

SharpeiClientPtr createSharpeiClient(http::HttpClientPtr httpClient, Settings settings,
    const RequestInfo& requestInfo);

SharpeiClientPtr makeCached(SharpeiClientPtr component,
        std::chrono::steady_clock::duration ttl = std::chrono::steady_clock::duration::max());

} // namespace
} // namespace
