#pragma once

namespace sharpei {
namespace client {
template <typename T, typename Now = decltype(&std::chrono::steady_clock::now)>
class Cache {
public:
    struct Data {
        Data(T value, std::chrono::steady_clock::time_point last_update_time)
            : value_(std::move(value))
            , last_update_time_(std::move(last_update_time))
        {}
        T value_;
        const std::chrono::steady_clock::time_point last_update_time_;
    };

    explicit Cache(const std::chrono::steady_clock::duration& ttl, Now now = &std::chrono::steady_clock::now, std::shared_ptr<Data> data = nullptr)
        : data_(data), now_(now), ttl_(ttl)
    {}

    void set(T value) {
        auto new_data = std::make_shared<const Data>(value, now_());
        std::atomic_store_explicit(&data_, new_data, std::memory_order_release);
    }

    std::shared_ptr<const Data> get() const {
        return std::atomic_load_explicit(&data_, std::memory_order_acquire);
    }

    inline bool isDataValid(std::shared_ptr<const Data> dataPtr) {
        return dataPtr && (now_() - dataPtr->last_update_time_) < ttl_;
    }

private:
    std::shared_ptr<const Data> data_;
    Now now_;
    const std::chrono::steady_clock::duration ttl_;
};

}
}
