#pragma once

#include <sharpei_client/sharpei_client.h>
#include <mail/sharpei_client/src/cache.h>

namespace sharpei {
namespace client {

template<typename Now = decltype(&std::chrono::steady_clock::now)>
class CachedSharpeiClient: public SharpeiClient,
                           public std::enable_shared_from_this<CachedSharpeiClient<Now>> {
public:
    using std::enable_shared_from_this<CachedSharpeiClient<Now>>::shared_from_this;

    CachedSharpeiClient(SharpeiClientPtr component,
            std::chrono::steady_clock::duration cacheTtl = std::chrono::steady_clock::duration::max(),
            Now now = &std::chrono::steady_clock::now)
            : connInfoCache(cacheTtl, now), deletedConnInfoCache(cacheTtl, now), orgConnInfoCache(cacheTtl, now),
             statCache(cacheTtl, now), statByIdCache(cacheTtl, now), component(component) {}

    void asyncGetConnInfo(const ResolveParams& params, AsyncHandler handler) const override {
        asyncGetConnInfoImpl(params, std::move(handler), connInfoCache,
                [self = shared_from_this()](const auto& p, auto h) {
                    self->component->asyncGetConnInfo(p, std::move(h));
                });
    }

    void asyncGetDeletedConnInfo(const ResolveParams& params, AsyncHandler handler) const override {
        asyncGetConnInfoImpl(params, std::move(handler), deletedConnInfoCache,
                [self = shared_from_this()](const auto& p, auto h) {
                    self->component->asyncGetDeletedConnInfo(p, std::move(h));
                });
    }

    void asyncGetOrgConnInfo(const ResolveParams& params, AsyncHandler handler) const override {
        asyncGetConnInfoImpl(params, std::move(handler), orgConnInfoCache,
                [self = shared_from_this()](const auto& p, auto h) {
                    self->component->asyncGetOrgConnInfo(p, std::move(h));
                });
    }

    void asyncStat(AsyncMapHandler handler) const override {
        auto statPtr = statCache.get();
        if (statCache.isDataValid(statPtr)) {
            handler(boost::system::error_code(), statPtr->value_);
            return;
        }

        const auto self = shared_from_this();
        const auto hook = [=](const ErrorCode& error, MapShard stat) {
            if (!error) {
                self->statCache.set(stat);
            }
            handler(error, std::move(stat));
        };
        component->asyncStat(std::move(hook));
    }

    void asyncStatById(const Shard::Id& id, AsyncHandler handler) const override {
        auto statPtr = statByIdCache.get();
        if (statByIdCache.isDataValid(statPtr) && statPtr->value_.id == id) {
            handler(boost::system::error_code(), statPtr->value_);
            return;
        }

        const auto self = shared_from_this();
        const auto hook = [=](const ErrorCode& error, Shard shard) {
            if (!error) {
                self->statByIdCache.set(shard);
            }
            handler(error, std::move(shard));
        };
        component->asyncStatById(id, std::move(hook));
    }

private:
    template<typename ComponentMethod>
    void asyncGetConnInfoImpl(const ResolveParams& params, AsyncHandler handler,
            Cache<ShardWithUid, Now>& cache, ComponentMethod componentMethod) const {
        if (!params.force) {
            auto shardWithUidPtr = cache.get();
            if (cache.isDataValid(shardWithUidPtr) && shardWithUidPtr->value_.uid == params.uid) {
                handler(boost::system::error_code(), shardWithUidPtr->value_.shard);
                return;
            }
        }

        auto self = shared_from_this();
        auto hook = [&cache, params, handler, self](const ErrorCode& error, Shard shard) mutable {
            if (!error) {
                cache.set({params.uid, shard});
            }
            handler(error, std::move(shard));
        };

        componentMethod(params, std::move(hook));
    }

    mutable Cache<ShardWithUid, Now> connInfoCache;
    mutable Cache<ShardWithUid, Now> deletedConnInfoCache;
    mutable Cache<ShardWithUid, Now> orgConnInfoCache;
    mutable Cache<MapShard, Now> statCache;
    mutable Cache<Shard, Now> statByIdCache;

    SharpeiClientPtr component;
};

SharpeiClientPtr makeCached(SharpeiClientPtr component, std::chrono::steady_clock::duration ttl) {
    return std::make_shared<CachedSharpeiClient<>>(component, ttl);
}

}
}
