#include <sharpei_client/sharpei_client.h>
#include <sharpei_client/reflection.h>
#include <yamail/data/deserialization/json_reader.h>
#include <boost/optional.hpp>
#include <sstream>
#include <mutex>
#include <unordered_map>
#include <sharpei_client/internal/http_client.h>

namespace sharpei {
namespace client {

std::string Shard::Database::Address::toString() const {
    std::ostringstream out;
    out << "host=" << host << " "
        << "port=" << port << " "
        << "dbname=" << dbname;
    return out.str();
}

std::string to_string(Mode type) {
    switch(type) {
        case Mode::WriteOnly: return "write_only";
        case Mode::WriteRead: return "write_read";
        case Mode::ReadWrite: return "read_write";
        case Mode::ReadOnly: return "read_only";
        case Mode::All: return "all";
        default:
            throw std::logic_error("Unknown Mode value" + std::to_string(int(type)));
    }
}

class RetryContext {
public:
    RetryContext()
        : count_(0)
    {}

    RetryContext(unsigned count, ErrorCode lastError)
        : count_(count), lastError_(lastError)
    {}

    RetryContext increment() const {
        return RetryContext(count_+1, lastError_);
    }

    unsigned count() const {
        return count_;
    }

    RetryContext lastError(ErrorCode error) const {
        return RetryContext(count_, error);
    }

    RetryContext lastError(Errors error) const {
        return lastError(makeErrorCode(error));
    }

    const ErrorCode& lastError() const {
        return lastError_;
    }

private:
    unsigned count_;
    ErrorCode lastError_;
};

class SharpeiClientImpl: public SharpeiClient,
        public std::enable_shared_from_this<SharpeiClientImpl> {
public:
    SharpeiClientImpl(http::HttpClientPtr httpClient, Settings settings, RequestInfo requestInfo)
        : httpClient_{std::move(httpClient)}, settings_{std::move(settings)}, requestInfo_{std::move(requestInfo)}
    {}

    void asyncGetConnInfo(const ResolveParams& params, AsyncHandler handler) const override {
        asyncGetConnInfoImpl(createConnInfoHttpArguments(params), std::move(handler), RetryContext(),
            {"/conninfo"});
    }

    void asyncGetDeletedConnInfo(const ResolveParams& params, AsyncHandler handler) const override {
        asyncGetConnInfoImpl(createConnInfoHttpArguments(params), std::move(handler), RetryContext(),
            {"/deleted_conninfo"});
    }

    void asyncGetOrgConnInfo(const ResolveParams& params, AsyncHandler handler) const override {
        asyncGetConnInfoImpl(createOrgConnInfoHttpArguments(params), std::move(handler), RetryContext(),
            {"/org_conninfo"});
    }

    void asyncStat(AsyncMapHandler handler) const override {
        asyncStat(std::move(handler), RetryContext());
    }

    void asyncStatById(const Shard::Id &id, AsyncHandler handler) const override {
        asyncStat([id, h = std::move(handler)](const ErrorCode& ec, MapShard shards) {
            if (ec) {
                h(ec, Shard{});
            } else if (shards.size() == 1 && shards.begin()->first == id) {
                h(ec, shards.begin()->second);
            } else {
                h(ErrorCode(Errors::ShardNotFound), Shard{});
            }
        }, RetryContext(), id);
    }

private:
    void asyncGetConnInfoImpl(http::Arguments args, AsyncHandler handler,
                              RetryContext retryCtx, std::string endpoint) const {
        if (stopOrIncrementRetries<AsyncHandler>(retryCtx, handler)) {
            return;
        }
        const http::Headers headers = createHttpHeaders(requestInfo_);
        auto thiz = shared_from_this();

        auto responseHandler = [=] (const ErrorCode& error, const http::Response& response) {
            ErrorCode resError;
            Shard res;
            try {
                if (error) {
                    thiz->settings_.onHttpError(error, "error from http client: " + error.message());
                    thiz->asyncGetConnInfoImpl(args, handler, retryCtx.lastError(error), endpoint);
                    return;
                } else if (thiz->isHttpCodeSuccess(response.code)) {
                    res = yamail::data::deserialization::fromJson<Shard>(response.body);
                } else if (thiz->isHttpCodeServerError(response.code)) {
                    thiz->asyncGetConnInfoImpl(args, handler, retryCtx.lastError(Errors::SharpeiError), endpoint);
                    return;
                } else if (response.code == 404) {
                    resError = makeErrorCode(Errors::UidNotFound,
                        "Sharpei service responded with 404");
                } else {
                    resError = makeErrorCode(Errors::HttpCode,
                        "Sharpei service responded with error code " + std::to_string(response.code));
                }
            } catch (const std::exception& e) {
                resError = makeErrorCode(Errors::Exception, e.what());
            }
            handler(resError, res);
        };
        try {
            httpClient_->aget(settings_.sharpeiAddress, settings_.timeout, endpoint, args,
                headers, responseHandler, settings_.keepAlive, requestInfo_.uniqId);
        } catch (const std::exception& e) {
            settings_.onHttpException(e);
            asyncGetConnInfoImpl(args, handler, retryCtx.lastError(Errors::Exception), endpoint);
        }
    }

    void asyncStat(AsyncMapHandler handler, RetryContext retryCtx, std::optional<Shard::Id> shardId = std::nullopt) const {
        if (stopOrIncrementRetries<AsyncMapHandler>(retryCtx, handler)) {
            return;
        }
        auto thiz = shared_from_this();

        auto responseHandler = [=] (const ErrorCode& error, const http::Response& response) {
            ErrorCode resError;
            MapShard res;
            try {
                if (error) {
                    thiz->settings_.onHttpError(error, "error from http client: " + error.message());
                    thiz->asyncStat(handler, retryCtx.lastError(error), shardId);
                    return;
                } else if (thiz->isHttpCodeSuccess(response.code)) {
                    res = yamail::data::deserialization::fromJson<MapShard>(response.body);
                } else if (response.code == 404) {
                    resError = ErrorCode(Errors::ShardNotFound);
                } else {
                    thiz->asyncStat(handler, retryCtx.lastError(Errors::SharpeiError), shardId);
                    return;
                }
            }  catch (const std::exception& e) {
                resError = makeErrorCode(Errors::Exception, e.what());
            }
            handler(resError, res);
        };
        http::Arguments args;
        if (shardId) {
            args["shard_id"] = { shardId.value() };
        }
        try {
            httpClient_->aget(settings_.sharpeiAddress, settings_.timeout, "/v3/stat",
                args, http::Headers(), responseHandler, settings_.keepAlive, "");
        } catch (const std::exception& e) {
            settings_.onHttpException(e);
            asyncStat(handler, retryCtx.lastError(Errors::Exception), shardId);
        }
    }

    template <typename Handler>
    bool stopOrIncrementRetries(RetryContext& retryCtx, Handler handler) const {
        if (retryCtx.count() >= settings_.retries) {
            try {
                handler(retryCtx.lastError(), typename Handler::second_argument_type());
            } catch (...) {}
            return true;
        } else {
            retryCtx = retryCtx.increment();
            return false;
        }
    }

    static http::Arguments createConnInfoHttpArguments(const ResolveParams& params) {
        http::Arguments args;
        args["uid"].push_back(params.uid);
        args["mode"].push_back(to_string(params.mode));
        if (params.force) {
            args["force"].push_back("1");
        }
        return args;
    }

    static http::Arguments createOrgConnInfoHttpArguments(const ResolveParams& params) {
        http::Arguments args;
        args["org_id"].push_back(params.uid);
        return args;
    }

    static http::Headers createHttpHeaders(const RequestInfo& requestInfo) {
        http::Headers res;
        res["X-Yandex-ClientType"].push_back(requestInfo.clientType);
        if (!requestInfo.connectionId.empty()) {
            res["X-Yandex-Session-Key"].push_back(requestInfo.connectionId);
        }
        if (!requestInfo.userIp.empty()) {
            res["X-Real-IP"].push_back(requestInfo.userIp);
        }
        if (!requestInfo.requestId.empty()) {
            res["X-Request-Id"].push_back(requestInfo.requestId);
        }
        return res;
    }

    static bool isHttpCodeSuccess(unsigned code) {
        return code / 100 == 2;
    }

    static bool isHttpCodeServerError(unsigned code) {
        return code / 100 == 5;
    }

    http::HttpClientPtr httpClient_;
    Settings settings_;
    RequestInfo requestInfo_;
};

SharpeiClientPtr createSharpeiClient(Settings settings, const RequestInfo& requestInfo) {
    if (!settings.httpClient) {
        throw std::logic_error("httpClient settings parameter is empty");
    }
    settings.retries = 1;
    auto httpClient = std::make_shared<sharpei::internal::DefaultHttpClient>(*settings.httpClient);
    return std::make_shared<SharpeiClientImpl>(std::move(httpClient), std::move(settings), requestInfo);
}

SharpeiClientPtr createSharpeiClient(http::HttpClientPtr httpClient, Settings settings,
        const RequestInfo& requestInfo) {
    if (!httpClient || settings.httpClient) {
        return createSharpeiClient(std::move(settings), requestInfo);
    }
    if (!httpClient) {
        throw std::logic_error("httpClient is empty");
    }
    return std::make_shared<SharpeiClientImpl>(std::move(httpClient), std::move(settings), requestInfo);
}

}
}
