from ymod_python_sys import log, severity_level
import json
import vobject
import re


# vcard fields
CONTACT_ID = "contact_id"
TYPE = "type"
VCARD = "vcard"
VCARD_NAME = "n"
VCARD_FULL_NAME = "fn"
VCARD_EMAIL = "email"
VCARD_TEL = "tel"
VCARD_NOTE = "note"
VCARD_EVENT_TYPES = {"birthday": "bday", "anniversary": "x-anniversary", "evolution-anniversary": "x-evolution-anniversary"}
VCARD_ORGANIZATION = "org"
VCARD_TITLE = "title"
VCARD_NICK_NAME = "nickname"
VCARD_ADDRESS = "adr"
VCARD_URL = "url"
VCARD_UID = "uid"
VCARD_PHOTO = "photo"
VCARD_SOCIAL_PROFILE = "x-socialprofile"
VCARD_MESSENGERS = "impp"
VCARD_MESSENGERS_SERVICE_TYPE = "x-service-type"
VCARD_ADDITIONAL_MESSENGERS = {"ICQ": "x-icq", "skype": "x-skype"}
VCARD_ADDITIONAL_SOCIAL_PROFILE = {"jabber": "x-jabber", "twitter": "x-twitter"}
VCARD_TZ = "tz"
NAMES = "names"
NAMES_PREFIX = "prefix"
NAMES_SUFFIX = "suffix"
NAMES_FIRST = "first"
NAMES_MIDDLE = "middle"
NAMES_LAST = "last"
EMAIL = "email"
EMAILS = "emails"
EMAILS_TYPE = ["WORK", "HOME", "MOBILE", "CUSTOM", "OTHER", "work", "home", "mobile", "custom", "other"]
PHONE = "telephone_number"
PHONES = "telephone_numbers"
PHONES_TYPES = ["WORK", "HOME", "CELL", "home", "work", "cell"]
NOTES = "notes"
EVENTS = "events"
YEAR = "year"
MONTH = "month"
DAY = "day"
ORGANIZATIONS = "organizations"
ORGANIZATIONS_COMPANY = "company"
ORGANIZATIONS_DEPARTAMENT = "department"
ORGANIZATIONS_TITLE = "title"
ORGANIZATIONS_SUMMARY = "summary"
NICK_NAMES = "nicknames"
ADDRESSES = "addresses"
ADDRESSES_STREET = "street"
ADDRESSES_CITY = "city"
ADDRESSES_REGION = "region"
ADDRESSES_COUNTRY = "country"
ADDRESSES_POSTAL_CODE = "postal_code"
ADDRESSES_POST_BOX = "post_office_box"
ADDRESSES_EXTENDED = "extended"
ADDRESSES_TYPE = ["WORK", "HOME", "home", "work"]
WEBSITES = "websites"
WEBSITES_URL = "url"
UIDS = "vcard_uids"
PHOTOS = "photos"
SOCIAL_PROFILES = "social_profiles"
SOCIAL_PROFILES_PROFILE = "profile"
SOCIAL_TYPE_FACEBOOK = 'fb'
SOCIAL_TYPE_VKONTAKTE = 'vk'
SOCIAL_TYPE_TWITTER = 'tw'
SOCIAL_TYPE_LINKEDIN = 'linkedin'
SOCIAL_PROFILES_TYPES = {
    SOCIAL_TYPE_FACEBOOK: ('facebook', 'http://facebook.com/%s'),
    SOCIAL_TYPE_VKONTAKTE: ('vkontakte', 'http://vk.com/%s'),
    SOCIAL_TYPE_TWITTER: ('twitter', 'http://twitter.com/%s'),
    SOCIAL_TYPE_LINKEDIN: ('linkedin', 'http://linkedin.com/%s'),
}
INSTANT_MESSENGERS = "instant_messengers"
INSTANT_MESSENGERS_SERVICE_ID = "service_id"
INSTANT_MESSENGERS_SERVICE_TYPE = "service_type"
INSTANT_MESSENGERS_PROTOCOL = "protocol"
INSTANT_MESSENGERS_X_SERVICE_TYPE = "X-SERVICE-TYPE"
CONTACT_TYPES = ["WORK", "HOME", "work", "home"]
TZS = "tzs"
DESCRIPTION = "description"


def makeFullName(name):
    return u" ".join(filter(None, [name.get(a) for a in (
        NAMES_PREFIX,
        NAMES_FIRST,
        NAMES_MIDDLE,
        NAMES_LAST,
        NAMES_SUFFIX)]))


def toVcardNames(contact, vobj):
    if len(contact[NAMES]) < 1:
        vobj.add(VCARD_FULL_NAME).value = ""
        vobj.add(VCARD_NAME).value = vobject.vcard.Name("")
        return

    # because vcard can has only single name
    name = contact[NAMES][0]
    vobj.add(VCARD_NAME).value = vobject.vcard.Name(
        family=name.get(NAMES_LAST, ""),
        given=name.get(NAMES_FIRST, ""),
        additional=name.get(NAMES_MIDDLE, ""),
        prefix=name.get(NAMES_PREFIX, ""),
        suffix=name.get(NAMES_SUFFIX, ""))
    vobj.add(VCARD_FULL_NAME).value = makeFullName(name)


def toVcardEmails(contact, vobj):
    for email in contact[EMAILS]:
        if EMAIL not in email:
            continue
        vcard_email = vobj.add(VCARD_EMAIL)
        vcard_email.value = email[EMAIL]
        if TYPE in email:
            for emailType in email[TYPE]:
                if emailType in EMAILS_TYPE:
                    vcard_email.type_param = emailType


def toVcardPhones(contact, vobj):
    for phone in contact[PHONES]:
        if PHONE not in phone:
            continue
        vcard_phone = vobj.add(VCARD_TEL)
        vcard_phone.value = phone[PHONE]
        if TYPE in phone:
            for phoneType in phone[TYPE]:
                if phoneType in PHONES_TYPES:
                    vcard_phone.type_param = phoneType


def toVcardNotes(contact, vobj):
    if NOTES in contact:
        for note in contact[NOTES]:
            vobj.add(VCARD_NOTE).value = note
    if DESCRIPTION in contact:
        description = contact[DESCRIPTION]
        if NOTES not in contact or description not in contact[NOTES]:
            vobj.add(VCARD_NOTE).value = description


def toVcardEvents(contact, vobj):
    for event in contact[EVENTS]:
        if TYPE in event:
            for eventType in event[TYPE]:
                if eventType in VCARD_EVENT_TYPES:
                    if MONTH in event and DAY in event and YEAR not in event:
                        # allow vCard 4 (RFC 6350 section 4.3.1)
                        vobj.add(VCARD_EVENT_TYPES[eventType]).value = "--%02d%02d" % (event[MONTH], event[DAY])
                    else:
                        month = event.get(MONTH, 0)
                        day = event.get(DAY, 0)
                        year = event.get(YEAR, 0)
                        vobj.add(VCARD_EVENT_TYPES[eventType]).value = "%04d%02d%02d" % (year, month, day)


def toVcardOrganizations(contact, vobj):
    for org in contact[ORGANIZATIONS]:
        if ORGANIZATIONS_COMPANY in org:
            vcard_org = vobj.add(VCARD_ORGANIZATION)
            vcard_org.value = [org[ORGANIZATIONS_COMPANY], org.get(ORGANIZATIONS_DEPARTAMENT, ""), org.get(ORGANIZATIONS_SUMMARY, ""), org.get(ORGANIZATIONS_TITLE, "")]
            if TYPE in org:
                vcard_org.type_param = org[TYPE]
        if ORGANIZATIONS_TITLE in org:
            vobj.add(VCARD_TITLE).value = org[ORGANIZATIONS_TITLE]


def toVcardNickNames(contact, vobj):
    for nick in contact[NICK_NAMES]:
        vobj.add(VCARD_NICK_NAME).value = nick


def toVcardAddresses(contact, vobj):
    for address in contact[ADDRESSES]:
        vcard_address = vobj.add(VCARD_ADDRESS)
        vcard_address.value = vobject.vcard.Address(
            street=address.get(ADDRESSES_STREET, ""),
            city=address.get(ADDRESSES_CITY, ""),
            region=address.get(ADDRESSES_REGION, ""),
            code=address.get(ADDRESSES_POSTAL_CODE, ""),
            country=address.get(ADDRESSES_COUNTRY, ""),
            box=address.get(ADDRESSES_POST_BOX, ""),
            extended=address.get(ADDRESSES_EXTENDED, ""),
        )
        if TYPE in address:
            for addressType in address[TYPE]:
                if addressType in ADDRESSES_TYPE:
                    vcard_address.type_param = addressType


def toVcardWebSites(contacts, vobj):
    for site in contacts[WEBSITES]:
        if WEBSITES_URL not in site:
            continue
        vobj.add(VCARD_URL).value = site[WEBSITES_URL]


def toVcardUids(contacts, vobj):
    for uid in contacts[UIDS]:
        vobj.add(VCARD_UID).value = uid


def toVcardPhotos(contacts, vobj):
    # must be data binary but we have url from db
    pass


def toVcardSocialProfile(contacts, vobj):
    for profile in contacts[SOCIAL_PROFILES]:
        vcard_profile = vobj.add(VCARD_SOCIAL_PROFILE)
        if TYPE in profile:
            if len(profile[TYPE]) > 0:
                vcard_profile.type_param = profile[TYPE][0]
                if profile[TYPE][0] in VCARD_ADDITIONAL_SOCIAL_PROFILE:
                    vcard_contact = vobj.add(VCARD_ADDITIONAL_SOCIAL_PROFILE[profile[TYPE][0]])
                    vcard_contact.value = profile[SOCIAL_PROFILES_PROFILE]
                    if (len(profile[TYPE]) > 1):
                        vcard_contact.type_param = profile[TYPE][1:]
            if SOCIAL_PROFILES_PROFILE not in profile:
                continue
            vcard_profile.value = profile[SOCIAL_PROFILES_PROFILE]


def toVcardInstantMessengers(contacts, vobj):
    for messenger in contacts[INSTANT_MESSENGERS]:
        if INSTANT_MESSENGERS_SERVICE_ID not in messenger:
            continue

        if INSTANT_MESSENGERS_SERVICE_TYPE in messenger:
            if messenger[INSTANT_MESSENGERS_SERVICE_TYPE] in VCARD_ADDITIONAL_MESSENGERS:
                vcard_contact = vobj.add(VCARD_ADDITIONAL_MESSENGERS[messenger[INSTANT_MESSENGERS_SERVICE_TYPE]])
                if TYPE in messenger:
                    vcard_contact.type_param = messenger[TYPE]
                vcard_contact.value = messenger[INSTANT_MESSENGERS_SERVICE_ID]

        vcard_impp = vobj.add(VCARD_MESSENGERS)
        if INSTANT_MESSENGERS_SERVICE_TYPE in messenger:
            vcard_impp.params[VCARD_MESSENGERS_SERVICE_TYPE] = [messenger[INSTANT_MESSENGERS_SERVICE_TYPE]]
        if INSTANT_MESSENGERS_PROTOCOL in messenger:
            vcard_impp.value = "%s:%s" % (
                messenger[INSTANT_MESSENGERS_PROTOCOL],
                messenger[INSTANT_MESSENGERS_SERVICE_ID])
        else:
            vcard_impp.value = messenger[INSTANT_MESSENGERS_SERVICE_ID]
        if TYPE in messenger:
            vcard_impp.type_param = messenger[TYPE]


def toVcardTimeZones(contact, vobj):
    for tz in contact[TZS]:
        vobj.add(VCARD_TZ).value = tz


# Not have in abook directory_entries, description
toVcardFields = {
    NAMES: toVcardNames,
    EMAILS: toVcardEmails,
    PHONES: toVcardPhones,
    EVENTS: toVcardEvents,
    ORGANIZATIONS: toVcardOrganizations,
    NICK_NAMES: toVcardNickNames,
    ADDRESSES: toVcardAddresses,
    WEBSITES: toVcardWebSites,
    UIDS: toVcardUids,
    PHOTOS: toVcardPhotos,
    SOCIAL_PROFILES: toVcardSocialProfile,
    INSTANT_MESSENGERS: toVcardInstantMessengers,
    TZS: toVcardTimeZones,
}


def toVcard(contact):
    vobj = vobject.vCard()
    if NAMES not in contact:
        vobj.add(VCARD_FULL_NAME).value = ""
        vobj.add(VCARD_NAME).value = vobject.vcard.Name("")
    for field in toVcardFields:
        if field in contact:
            encoder = toVcardFields.get(field)
            encoder(contact, vobj)
    toVcardNotes(contact, vobj)
    return vobj.serialize()


def transformToVcard(uid, contactsStr):
    try:
        result = dict()
        contacts = json.loads(contactsStr)
        for contactId, contact in contacts.iteritems():
            if UIDS not in contact:
                contact[UIDS] = ["YAAB" + "-" + uid + "-" + contactId]
            result[contactId] = toVcard(contact)
        return bytes(json.dumps(result))
    except BaseException as error:
        log(severity_level.error, "uid={}, message=[Python transformToVcard exception occurred: {}]".format(uid, error))


def fromVcardNames(vobj, contact):
    for name in vobj.contents[VCARD_NAME]:
        jsonName = dict()
        jsonName[NAMES_LAST] = name.value.family
        jsonName[NAMES_FIRST] = name.value.given
        jsonName[NAMES_MIDDLE] = name.value.additional
        jsonName[NAMES_PREFIX] = name.value.prefix
        jsonName[NAMES_SUFFIX] = name.value.suffix
        if NAMES not in contact:
            contact[NAMES] = list()
        contact[NAMES].append(jsonName)


def fromVcardEmails(vobj, contact):
    for email in vobj.contents[VCARD_EMAIL]:
        if EMAILS not in contact:
            contact[EMAILS] = list()
        resultEmail = {EMAIL: email.value}
        if hasattr(email, 'type_param'):
            resultEmail[TYPE] = [email.type_param]
        contact[EMAILS].append(resultEmail)


def fromVcardPhones(vobj, contact):
    for phone in vobj.contents[VCARD_TEL]:
        if PHONES not in contact:
            contact[PHONES] = list()
        resultPhone = {PHONE: phone.value}
        if hasattr(phone, 'type_param'):
            resultPhone[TYPE] = [phone.type_param]
        contact[PHONES].append(resultPhone)


def fromVcardNotes(vobj, contact):
    for note in vobj.contents[VCARD_NOTE]:
        if NOTES not in contact:
            contact[NOTES] = list()
        contact[NOTES].append(note.value)
    if NOTES in contact:
        contact[DESCRIPTION] = contact[NOTES][0]


def parseDate(event):
    result = {YEAR: 0, MONTH: 0, DAY: 0}
    regexRfc = re.compile("^(\\d\\d\\d\\d)-?(\\d\\d)-?(\\d\\d)$")
    regexWithoutYear = re.compile("^--(\\d\\d)(\\d\\d)$")
    matched = regexRfc.match(event)
    if matched is not None:
        result[YEAR] = int(matched.groups()[0])
        result[MONTH] = int(matched.groups()[1])
        result[DAY] = int(matched.groups()[2])

    else:
        matched = regexWithoutYear.match(event)
        if matched is not None:
            result[MONTH] = int(matched.groups()[0])
            result[DAY] = int(matched.groups()[1])
    return result


def fromVcardEvents(vobj, contact):
    for event_type in VCARD_EVENT_TYPES:
        if VCARD_EVENT_TYPES[event_type] in vobj.contents:
            for event in vobj.contents[VCARD_EVENT_TYPES[event_type]]:
                jsonBday = parseDate(event.value)
                if EVENTS not in contact:
                    contact[EVENTS] = list()
                contact[EVENTS].append({
                    YEAR: jsonBday[YEAR],
                    MONTH: jsonBday[MONTH],
                    DAY: jsonBday[DAY],
                    TYPE: [event_type],
                })


def fromVcardOrganizations(vobj, contact):
    for org in vobj.contents[VCARD_ORGANIZATION]:
        if ORGANIZATIONS not in contact:
            contact[ORGANIZATIONS] = list()
        len_org = len(org.value)
        organization = {ORGANIZATIONS_COMPANY: org.value[0]}
        if len_org > 1:
            organization[ORGANIZATIONS_DEPARTAMENT] = org.value[1]
        if len_org > 2:
            organization[ORGANIZATIONS_SUMMARY] = org.value[2]
        if len_org > 3:
            organization[ORGANIZATIONS_TITLE] = org.value[3]
        if hasattr(org, 'type_param'):
            organization[TYPE] = [org.type_param]
        contact[ORGANIZATIONS].append(organization)

    # title separated from org dield in vcard. Missing this field
    # for title in vobj.contents[VCARD_TITLE]:


def fromVcardNickNames(vobj, contact):
    for nick in vobj.contents[VCARD_NICK_NAME]:
        if NICK_NAMES not in contact:
            contact[NICK_NAMES] = list()
        contact[NICK_NAMES].append(nick.value)


def fromVcardAddresses(vobj, contact):
    for address in vobj.contents[VCARD_ADDRESS]:
        jsonAddress = dict()
        jsonAddress[ADDRESSES_STREET] = address.value.street
        jsonAddress[ADDRESSES_CITY] = address.value.city
        jsonAddress[ADDRESSES_REGION] = address.value.region
        jsonAddress[ADDRESSES_POSTAL_CODE] = address.value.code
        jsonAddress[ADDRESSES_POST_BOX] = address.value.box
        jsonAddress[ADDRESSES_COUNTRY] = address.value.country
        jsonAddress[ADDRESSES_EXTENDED] = address.value.extended
        if hasattr(address, 'type_param'):
            jsonAddress[TYPE] = [address.type_param]
        if ADDRESSES not in contact:
            contact[ADDRESSES] = list()
        contact[ADDRESSES].append(jsonAddress)


def fromVcardWebSites(vobj, contact):
    for url in vobj.contents[VCARD_URL]:
        if WEBSITES not in contact:
            contact[WEBSITES] = list()
        contact[WEBSITES].append({WEBSITES_URL: url.value})


def fromVcardUids(vobj, contact):
    for uid in vobj.contents[VCARD_UID]:
        if UIDS not in contact:
            contact[UIDS] = list()
        contact[UIDS].append(uid.value)


def fromVcardPhotos(vobj, contact):
    # must be url but we have data binary
    pass


def fromVcardSocialProfile(vobj, contact):
    for profile in vobj.contents[VCARD_SOCIAL_PROFILE]:
        if SOCIAL_PROFILES not in contact:
            contact[SOCIAL_PROFILES] = list()
        resultSocialProfile = {SOCIAL_PROFILES_PROFILE: profile.value}
        if hasattr(profile, 'type_param'):
            resultSocialProfile[TYPE] = [profile.type_param]
        contact[SOCIAL_PROFILES].append(resultSocialProfile)


def fromVcardInstantMessengers(vobj, contact):
    for messenger in vobj.contents[VCARD_MESSENGERS]:
        splitedProtocol = messenger.value.split(":")
        serviceType = messenger.params.get(INSTANT_MESSENGERS_X_SERVICE_TYPE, [""])
        if len(splitedProtocol) < 2:
            continue
        if INSTANT_MESSENGERS not in contact:
            contact[INSTANT_MESSENGERS] = list()
        contact[INSTANT_MESSENGERS].append({
            INSTANT_MESSENGERS_PROTOCOL: splitedProtocol[0],
            INSTANT_MESSENGERS_SERVICE_ID: splitedProtocol[1],
            INSTANT_MESSENGERS_SERVICE_TYPE: serviceType[0]
        })


def fromVcardAdditionalSocialProfile(vobj, contact):
    for social_profile_type in VCARD_ADDITIONAL_SOCIAL_PROFILE:
        if VCARD_ADDITIONAL_SOCIAL_PROFILE[social_profile_type] in vobj.contents:
            for social_profile in vobj.contents[VCARD_ADDITIONAL_SOCIAL_PROFILE[social_profile_type]]:
                if SOCIAL_PROFILES not in contact:
                    contact[SOCIAL_PROFILES] = list()
                resultSocialProfile = {SOCIAL_PROFILES_PROFILE: social_profile.value}
                resultSocialProfile[TYPE] = [social_profile_type]
                if hasattr(social_profile, 'type_param'):
                    resultSocialProfile[TYPE].append(social_profile.type_param)
                contact[SOCIAL_PROFILES].append(resultSocialProfile)


def fromVcardAdditionalInstantMessengers(vobj, contact):
    for messenger_type in VCARD_ADDITIONAL_MESSENGERS:
        if VCARD_ADDITIONAL_MESSENGERS[messenger_type] in vobj.contents:
            for messenger in vobj.contents[VCARD_ADDITIONAL_MESSENGERS[messenger_type]]:
                resultMessenger = {
                    INSTANT_MESSENGERS_SERVICE_ID: messenger.value,
                    INSTANT_MESSENGERS_SERVICE_TYPE: messenger_type
                }
                if hasattr(messenger, 'type_param'):
                    resultMessenger[TYPE] = messenger.type_param
                if INSTANT_MESSENGERS not in contact:
                    contact[INSTANT_MESSENGERS] = list()
                contact[INSTANT_MESSENGERS].append(resultMessenger)


def fromVcardTimeZones(vobj, contact):
    for tz in vobj.contents[VCARD_TZ]:
        if TZS not in contact:
            contact[TZS] = list()
        contact[TZS].append(tz.value)


fromVcardFields = {
    VCARD_NAME: fromVcardNames,
    VCARD_EMAIL: fromVcardEmails,
    VCARD_TEL: fromVcardPhones,
    VCARD_NOTE: fromVcardNotes,
    VCARD_ORGANIZATION: fromVcardOrganizations,
    VCARD_NICK_NAME: fromVcardNickNames,
    VCARD_ADDRESS: fromVcardAddresses,
    VCARD_URL: fromVcardWebSites,
    VCARD_UID: fromVcardUids,
    VCARD_PHOTO: fromVcardPhotos,
    VCARD_SOCIAL_PROFILE: fromVcardSocialProfile,
    VCARD_MESSENGERS: fromVcardInstantMessengers,
    VCARD_TZ: fromVcardTimeZones,
}


def fromVcard(vobj):
    contact = dict()
    for field in fromVcardFields:
        if field in vobj.contents:
            log(severity_level.error, field)
            decoder = fromVcardFields.get(field)
            decoder(vobj, contact)
    fromVcardEvents(vobj, contact)
    fromVcardAdditionalSocialProfile(vobj, contact)
    fromVcardAdditionalInstantMessengers(vobj, contact)
    return contact


def transformFromVcard(uid, VcardStr):
    try:
        vobj = vobject.readOne(VcardStr)
        return bytes(json.dumps(fromVcard(vobj)))
    except BaseException as error:
        log(severity_level.error, "uid={}, message=[Python transformFromVcard exception occurred: {}]".format(uid, error))


def vcardEncoder(uid, contacts):
    jsonContacts = json.loads(contacts)
    result = str()
    for contact in jsonContacts["contacts"]:
        vcard = contact["vcard"]
        if UIDS not in vcard:
            vcard[UIDS] = ["YAAB" + "-" + uid + "-" + str(contact["contact_id"])]
        result += toVcard(vcard)
    return result


def exportContacts(uid, contactsStr):
    try:
        return bytes(vcardEncoder(uid, contactsStr))
    except BaseException as error:
        log(severity_level.error, "uid={}, message=[Python exportContacts exception occurred: {}]".format(uid, error))


def vcardDecoder(contacts):
    vstream = vobject.readComponents(contacts, ignoreUnreadable=True)
    result = list()
    for vcard in vstream:
        result.append({"vcard": fromVcard(vcard)})
    return result


def importContacts(uid, contactsStr):
    try:
        return bytes(json.dumps(vcardDecoder(contactsStr)))
    except BaseException as error:
        log(severity_level.error, "uid={}, message=[Python importContacts exception occurred: {}]".format(uid, error))
