#pragma once

#include <algorithm>
#include <boost/fusion/adapted/struct/adapt_struct.hpp>

namespace sheltie::collie {

using ContactId = std::int64_t;
using Revision = std::int64_t;
using ListId = std::int64_t;
using TagId = std::int64_t;
using Tid = std::int64_t;

struct Name {
    std::optional<std::string> first;
    std::optional<std::string> middle;
    std::optional<std::string> last;
    std::optional<std::string> suffix;
    std::optional<std::string> prefix;
};

struct Email {
    std::optional<std::string> email;
    std::optional<std::vector<std::string>> type;
    std::optional<std::string> label;
};

struct InstantMessenger {
    std::optional<std::string> protocol;
    std::optional<std::vector<std::string>> type;
    std::optional<std::string> service_id;
    std::optional<std::string> service_type;
};

struct SocialProfile {
    std::optional<std::string> profile;
    std::optional<std::vector<std::string>> type;
    std::optional<std::string> label;
};

struct TelephoneNumber {
    std::optional<std::string> telephone_number;
    std::optional<std::string> additional;
    std::optional<std::vector<std::string>> type;
    std::optional<std::string> label;
};

struct DirectoryEntry {
    std::optional<std::int64_t> entry_id;
    std::optional<std::vector<std::string>> type;
};

struct Event {
    std::optional<std::int64_t> year;
    std::optional<std::int64_t> month;
    std::optional<std::int64_t> day;
    std::optional<std::string> label;
    std::optional<std::vector<std::string>> type;
};

struct Photo {
    std::optional<std::string> uri;
    std::optional<std::string> storage;
    std::optional<std::vector<std::string>> type;
};

struct Address {
    std::optional<std::string> extended;
    std::optional<std::string> post_office_box;
    std::optional<std::string> street;
    std::optional<std::string> city;
    std::optional<std::string> region;
    std::optional<std::string> postal_code;
    std::optional<std::string> country;
    std::optional<std::string> label;
    std::optional<std::vector<std::string>> type;
};

struct Organization {
    std::optional<std::string> company;
    std::optional<std::string> title;
    std::optional<std::string> summary;
    std::optional<std::string> department;
    std::optional<std::vector<std::string>> type;
};

struct Website {
    std::optional<std::string> url;
    std::optional<std::vector<std::string>> type;
    std::optional<std::string> label;
};

struct Vcard {
    std::optional<std::vector<Name>> names;
    std::optional<std::vector<Email>> emails;
    std::optional<std::vector<InstantMessenger>> instant_messengers;
    std::optional<std::vector<SocialProfile>> social_profiles;
    std::optional<std::vector<TelephoneNumber>> telephone_numbers;
    std::optional<std::vector<std::string>> vcard_uids;
    std::optional<std::vector<DirectoryEntry>> directory_entries;
    std::optional<std::vector<std::string>> notes;
    std::optional<std::string> description;
    std::optional<std::vector<Event>> events;
    std::optional<std::vector<Photo>> photos;
    std::optional<std::vector<Address>> addresses;
    std::optional<std::vector<Organization>> organizations;
    std::optional<std::vector<std::string>> nicknames;
    std::optional<std::vector<Website>> websites;
    std::optional<std::vector<std::string>> tzs;
};

template <typename Params>
inline void sortVcardField(Params& params) {
    if (params) {
        std::sort(params->begin(), params->end());
    }
}

inline void sortVcard(Vcard& vcard) {
    sortVcardField(vcard.names);
    sortVcardField(vcard.vcard_uids);
    sortVcardField(vcard.emails);
    sortVcardField(vcard.telephone_numbers);
}

struct AbookTypeEmail {
    std::optional<std::int64_t> id;
    std::optional<std::string> value;
    std::optional<std::vector<Tid>> tags;
};

}

BOOST_FUSION_ADAPT_STRUCT(sheltie::collie::Name,
    prefix,
    first,
    middle,
    last,
    suffix
)

BOOST_FUSION_ADAPT_STRUCT(sheltie::collie::Email,
    email,
    type,
    label
)

BOOST_FUSION_ADAPT_STRUCT(sheltie::collie::InstantMessenger,
    protocol,
    type,
    service_id,
    service_type
)

BOOST_FUSION_ADAPT_STRUCT(sheltie::collie::SocialProfile,
    profile,
    type,
    label
)

BOOST_FUSION_ADAPT_STRUCT(sheltie::collie::TelephoneNumber,
    telephone_number,
    additional,
    type,
    label
)

BOOST_FUSION_ADAPT_STRUCT(sheltie::collie::DirectoryEntry,
    entry_id,
    type
)

BOOST_FUSION_ADAPT_STRUCT(sheltie::collie::Event,
    year,
    month,
    day,
    label,
    type
)

BOOST_FUSION_ADAPT_STRUCT(sheltie::collie::Photo,
    uri,
    storage,
    type
)

BOOST_FUSION_ADAPT_STRUCT(sheltie::collie::Address,
    extended,
    post_office_box,
    street,
    city,
    region,
    postal_code,
    country,
    label,
    type
)

BOOST_FUSION_ADAPT_STRUCT(sheltie::collie::Organization,
    company,
    title,
    summary,
    department,
    type
)

BOOST_FUSION_ADAPT_STRUCT(sheltie::collie::Website,
    url,
    type,
    label
)

BOOST_FUSION_ADAPT_STRUCT(sheltie::collie::Vcard,
    names,
    emails,
    instant_messengers,
    social_profiles,
    telephone_numbers,
    vcard_uids,
    directory_entries,
    notes,
    description,
    events,
    photos,
    addresses,
    organizations,
    nicknames,
    websites,
    tzs
)

BOOST_FUSION_ADAPT_STRUCT(sheltie::collie::AbookTypeEmail,
    id,
    value,
    tags
)

namespace sheltie::collie {

static bool operator==(const Name& left, const Name& right) {
    return boost::fusion::operator==(left, right);
}

static bool operator<(const Name& left, const Name& right) {
    return std::tie(left.prefix, left.first, left.middle, left.last, left.suffix) <
        std::tie(right.prefix, right.first, right.middle, right.last, right.suffix);
}

static bool operator==(const Email& left, const Email& right) {
    return left.email == right.email;
}

static bool operator<(const Email& left, const Email& right) {
    return left.email < right.email;
}

static bool operator==(const TelephoneNumber& left, const TelephoneNumber& right) {
    return left.telephone_number == right.telephone_number;
}

static bool operator<(const TelephoneNumber& left, const TelephoneNumber& right) {
    return left.telephone_number < right.telephone_number;
}

static bool operator==(const Vcard& left, const Vcard& right) {
    return std::tie(left.names, left.emails, left.telephone_numbers, left.vcard_uids) ==
        std::tie(right.names, right.emails, right.telephone_numbers, right.vcard_uids);
}

}