#pragma once

#include <common/types.h>
#include <common/log.h>
#include <ymod_python/module.h>
#include <yplatform/module.h>
#include <yplatform/find.h>
#include <boost/asio.hpp>
#include <boost/python.hpp>

namespace sheltie::python {

struct Module : public yplatform::module {
    void init (const yplatform::ptree& /*config*/) {
        python = ymod_python::get_current_python();
    }

    void fini() {
        python.reset();
    }

    template <typename CompletionToken>
    auto transformToVcard(
        const std::string& uid,
        const std::string& contacts,
        CompletionToken&& token)
    {
        return runPython("transformToVcard", uid, contacts, std::forward<CompletionToken>(token));
    }

    template <typename CompletionToken>
    auto transformFromVcard(
        const std::string& uid,
        const std::string& contacts,
        CompletionToken&& token)
    {
        return runPython("transformFromVcard", uid, contacts, std::forward<CompletionToken>(token));
    }

    template <typename CompletionToken>
    auto importContacts(
        const std::string& uid,
        const std::string& contacts,
        CompletionToken&& token)
    {
        return runPython("importContacts", uid, contacts, std::forward<CompletionToken>(token));
    }

    template <typename CompletionToken>
    auto exportContacts(
        const std::string& uid,
        const std::string& contacts,
        CompletionToken&& token)
    {
        return runPython("exportContacts", uid, contacts, std::forward<CompletionToken>(token));
    }

    template <typename CompletionToken>
    auto runPython(
        const std::string& functionName,
        const std::string& uid,
        const std::string& request,
        CompletionToken&& token)
    {
        using Signature = void (boost::system::error_code, std::string);
        boost::asio::async_completion<CompletionToken, Signature> init(token);
        python->async_run([functionName, uid, request, handler=std::move(init.completion_handler)] () mutable {
            using namespace boost::python;
            object main = import("__main__");
            object global(main.attr("__dict__"));
            object transform = global[functionName];
            object objUid{uid};
            object objRequest{request};
            try {
                object result = transform(objUid, objRequest);
                auto result_ptr = result.ptr();
                if (!PyString_Check(result_ptr)) {
                    LOGDOG_(getLogger(), error, log::message="python cannot convert result to string");
                    return handler(boost::asio::error::operation_aborted, std::string());
                }
                auto bytes = PyString_AsString(result_ptr);
                handler(ErrorCode(), std::string{bytes});
            } catch (const error_already_set&) {
                PyErr_Clear();
                LOGDOG_(getLogger(), error, log::message="python interpreter exception occured");
                handler(boost::asio::error::operation_aborted, std::string());
            }
        });
        return init.result.get();
    }

    ymod_python::python_ptr python;
};

}
