#pragma once

#include <common/types.h>
#include <common/log.h>
#include <web/common.h>
#include <web/context.h>

#include <memory>

namespace sheltie::web {

template <typename TvmGuardPtr>
inline bool isTvmTicketValid(TvmGuardPtr guard, RequestPtr request, const std::string& uid) {
    auto result = guard->check(
        request->url.make_full_path(),
        uid,
        getOrNone(request->headers, tvm_guard::header::serviceTicket()),
        getOrNone(request->headers, tvm_guard::header::userTicket()));

    return result.action == tvm_guard::Action::accept;
}

struct CheckTvmOp {
    template <typename Operation, typename WebContextPtr, typename... Args>
    void operator()(YieldCtx yield, WebContextPtr webCtx, StreamPtr stream, const std::string& uid, Operation&& op, Args&&... args) {
        auto webServerCtx = stream->ctx();
        auto webServerRequest = stream->request();
        auto logger = getLogger(getUniqtId(webServerCtx), getRequestId(webServerRequest));
        try {
            if (isTvmTicketValid(webCtx->tvmGuard, webServerRequest, uid)) {
                op(yield, webCtx, logger, stream, uid, std::forward<Args>(args)...);
            } else {
                LOGDOG_(logger, error, log::uid=uid, log::message="bad tvm ticket");
                stream->result(ymod_webserver::codes::unauthorized, "bad tvm ticket");
            }
        } catch(const std::exception& e) {
            logException(logger, e);
            auto code = ymod_webserver::codes::internal_server_error;
            stream->result(code, ymod_webserver::codes::reason::get(code));
        }
    }
};

}
