#pragma once

#include <common/log.h>
#include <common/types.h>
#include <collie/request.h>
#include <web/common.h>
#include <web/context.h>

#include <sstream>

namespace sheltie::web {

struct ExportContactsOp {
    template <typename WebContextPtr>
    void operator()(
        YieldCtx yieldCtx,
        WebContextPtr webCtx,
        RequestLogger logger,
        StreamPtr stream,
        const std::string& uid)
    {
        auto webServerRequest = stream->request();
        auto webServerCtx = webServerRequest->ctx();
        auto HttpRequest = collie::makeGetContactsRequest(uid);
        try {
            ErrorCode ec;
            auto collieResponse = webCtx->collieClient->async_run(webServerCtx, HttpRequest, yieldCtx[ec]);
            std::string contactsResponse;
            if (!ec) {
                if (collieResponse.status != 200) {
                    LOGDOG_(
                        logger,
                        error,
                        log::uid=uid,
                        log::where_name=getUrlPath(webServerRequest),
                        log::http_status=collieResponse.status,
                        log::message=collieResponse.reason);
                    std::string reason = (collieResponse.status / 100 == 5) ? "export contacts error: " + collieResponse.reason : "get contacts error";
                    return stream->result(ymod_webserver::codes::code(collieResponse.status), reason);
                }
                contactsResponse = webCtx->pythonModule->exportContacts(uid, collieResponse.body, yieldCtx[ec]);
            }
            if (ec) {
                LOGDOG_(logger, error, log::uid=uid, log::where_name=getUrlPath(webServerRequest), log::error_code=ec);
                return stream->result(
                    ymod_webserver::codes::internal_server_error,
                    "export contacts error: " + ec.message());
            }
            stream->set_code(ymod_webserver::codes::ok);
            stream->result_body(contactsResponse);
        } catch (const std::exception& e) {
            logException(logger, e);
            auto code = ymod_webserver::codes::internal_server_error;
            stream->result(code, ymod_webserver::codes::reason::get(code));
        }
    }
};

}
