#pragma once

#include <common/log.h>
#include <common/types.h>
#include <web/common.h>
#include <web/context.h>
#include <web/import_contacts_result.h>
#include <python/module.h>
#include <collie/types/new_contact.h>
#include <collie/types/created_contacts.h>
#include <collie/types/existing_contacts.h>
#include <collie/request.h>
#include <yamail/data/serialization/yajl.h>
#include <yamail/data/deserialization/json_reader.h>

#include <sstream>

namespace sheltie::web {

inline std::vector<sheltie::collie::NewContact> deduplicateExistingContacts(
    std::vector<sheltie::collie::NewContact>& importedContacts,
    sheltie::collie::ExistingContacts& existingContacts)
{
    for (auto& contact : existingContacts.contacts) {
        collie::sortVcard(contact.vcard);
    }
    std::vector<sheltie::collie::NewContact> newContacts;
    newContacts.reserve(importedContacts.size());
    for (auto& importedContact : importedContacts) {
        collie::sortVcard(importedContact.vcard);

        auto it = std::find_if(
            existingContacts.contacts.begin(),
            existingContacts.contacts.end(),
            [importedContact] (const auto& existingConact) {
                return importedContact.vcard == existingConact.vcard;
            });
        if (it == existingContacts.contacts.end()) {
            newContacts.push_back(std::move(importedContact));
        }
    }
    return newContacts;
}

inline void makeOkResponse(StreamPtr stream, std::int64_t createdCount, std::int64_t skippedCount = 0) {
    ImportContactsResult result;
    result.rec_cnt = createdCount;
    result.rec_skipped = skippedCount;
    auto response = yamail::data::serialization::toJson(result);
    stream->set_code(ymod_webserver::codes::ok);
    stream->result_body(response);
}

struct ImportContactsOp {
    template <typename WebContextPtr>
    void operator()(
        YieldCtx yieldCtx,
        WebContextPtr webCtx,
        RequestLogger logger,
        StreamPtr stream,
        const std::string& uid)
    {
        using yamail::data::deserialization::fromJson;
        using yamail::data::serialization::toJson;
        auto webServerRequest = stream->request();
        auto webServerCtx = webServerRequest->ctx();
        try {
            ErrorCode ec;
            auto parsedContacts = webCtx->pythonModule->importContacts(
                uid, 
                std::string(webServerRequest->raw_body.begin(), webServerRequest->raw_body.end()), 
                yieldCtx[ec]);
            if (ec) {
                LOGDOG_(
                    logger,
                    error,
                    log::uid=uid,
                    log::where_name=getUrlPath(webServerRequest),
                    log::error_code=ec,
                    log::message="cannot transform to vcard");
                return stream->result(
                    ymod_webserver::codes::internal_server_error,
                    "import contacts error: " + ec.message());
            }
            auto importedContacts = fromJson<std::vector<sheltie::collie::NewContact>>(parsedContacts);
            if (importedContacts.empty()) {
                LOGDOG_(
                    logger,
                    warning,
                    log::uid=uid,
                    log::where_name=getUrlPath(webServerRequest),
                    log::message="empty contacts for import");
                return makeOkResponse(stream, 0);
            }
            auto getContactsRequest = collie::makeGetContactsRequest(uid);
            auto getContactsResponse = webCtx->collieClient->async_run(webServerCtx, getContactsRequest, yieldCtx[ec]);
            if (ec) {
                LOGDOG_(
                    logger,
                    error,
                    log::uid=uid,
                    log::where_name=getUrlPath(webServerRequest),
                    log::error_code=ec,
                    log::message="cannot get contacts");
                return stream->result(
                    ymod_webserver::codes::internal_server_error,
                    "import contacts error: " + ec.message());
            }
            if (getContactsResponse.status != 200) {
                LOGDOG_(
                    logger,
                    error,
                    log::uid=uid,
                    log::where_name=getUrlPath(webServerRequest),
                    log::http_status=getContactsResponse.status,
                    log::message="cannot get contacts reason=" + getContactsResponse.reason);
                std::string reason = (getContactsResponse.status / 100 == 5) ? "import contacts error: " + getContactsResponse.reason : "get contacts error";
                return stream->result(ymod_webserver::codes::code(getContactsResponse.status), reason);
            }
            auto existingContacts = fromJson<sheltie::collie::ExistingContacts>(getContactsResponse.body);
            auto newContacts = deduplicateExistingContacts(importedContacts, existingContacts);
            if (newContacts.empty()) {
                LOGDOG_(
                    logger,
                    error,
                    log::uid=uid,
                    log::where_name=getUrlPath(webServerRequest),
                    log::message="all contacts deduplicated");
                return makeOkResponse(stream, 0, importedContacts.size());
            }
            auto createContactsRequest = collie::makeCreateContactsRequest(uid, std::move(toJson(newContacts)));
            auto createContactsResponse = webCtx->collieClient->async_run(webServerCtx, createContactsRequest, yieldCtx[ec]);
            if (ec) {
                LOGDOG_(
                    logger,
                    error,
                    log::uid=uid,
                    log::where_name=getUrlPath(webServerRequest),
                    log::error_code=ec,
                    log::message="cannot create contacts");
                return stream->result(
                    ymod_webserver::codes::internal_server_error,
                    "import contacts error: " + ec.message());
            }
            if (createContactsResponse.status != 200) {
                LOGDOG_(
                    logger,
                    error,
                    log::uid=uid,
                    log::where_name=getUrlPath(webServerRequest),
                    log::http_status=createContactsResponse.status,
                    log::message="cannot create contacts reason=" + createContactsResponse.reason);
                std::string reason = (createContactsResponse.status / 100 == 5) ? "import contacts error: " + createContactsResponse.reason : "create contacts error";
                return stream->result(ymod_webserver::codes::code(createContactsResponse.status), reason);
            }
            auto createContactsResult = fromJson<sheltie::collie::CreatedContacts>(createContactsResponse.body);
            makeOkResponse(
                stream,
                createContactsResult.contact_ids.size(),
                importedContacts.size() - createContactsResult.contact_ids.size());
        } catch (const std::exception& e) {
            logException(logger, e);
            auto code = ymod_webserver::codes::internal_server_error;
            stream->result(code, ymod_webserver::codes::reason::get(code));
        }
    }
};

}
