#include <common/log.h>
#include <common/spawn.h>
#include <web/check_tvm.h>
#include <web/binded_spawn.h>
#include <web/common.h>
#include <web/context.h>
#include <web/module.h>
#include <web/handlers/ping.h>
#include <web/handlers/to_vcard_op.h>
#include <web/handlers/from_vcard_op.h>
#include <web/handlers/export_contacts_op.h>
#include <web/handlers/import_contacts_op.h>

namespace sheltie::web {

void Module::init (const yplatform::ptree& cfg) {
    auto reactor = yplatform::find_reactor(cfg.get<std::string>("reactor"));
    for (size_t i = 0; i < reactor->size(); ++i) {
        if ((*reactor)[i]->size() != 1)
            throw std::runtime_error("sheltie_web is optimized for single-thread reactors - set pool_count=N and io_threads=1");
    }

    auto tvm_module = yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>("tvm");
    auto tvmGuard = std::make_shared<TvmGuard>(tvm_guard::init(cfg.get_child("tvm_guard"), tvm_module));
    auto collieClient = yplatform::find<yhttp::cluster_client>("collie_client");
    auto pythonModule = yplatform::find<sheltie::python::Module, std::shared_ptr>("sheltie_python");
    auto webCtx = std::make_shared<WebContext>(
        WebContext{*reactor->io(),
        tvmGuard,
        collieClient,
        pythonModule});
    auto webserver = yplatform::find<ymod_webserver::server, std::shared_ptr>("web_server");

    using ymod_webserver::transformer;
    using ymod_webserver::argument;
    webserver->bind("", {"/ping"}, ping);
    webserver->bind("", {"/compat/colabook_import.json"},
        bindedSpawn(
            webCtx,
            CheckTvmOp{},
            ImportContactsOp{}
        ), transformer(argument<std::string>("uid")));
    webserver->bind("", {"/compat/colabook_export"},
        bindedSpawn(
            webCtx,
            CheckTvmOp{},
            ExportContactsOp{}
        ), transformer(argument<std::string>("uid")));
    webserver->bind("", {"/v1/to_vcard"}, bindedSpawn(webCtx, CheckTvmOp{}, ToVcardOp{}), transformer(argument<std::string>("uid")));
    webserver->bind("", {"/v1/from_vcard"}, bindedSpawn(webCtx, CheckTvmOp{}, FromVcardOp{}), transformer(argument<std::string>("uid")));
}

}

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(sheltie::web::Module)
