from .base import BaseClient


class MaildbClient(BaseClient):
    def cleanup_doomed(self):
        """
        Deletes doomed messages for all shards
        """
        return self.get('/cleanup_doomed')

    def close_for_load(self, max_transfer_ratio=None, max_registration_ratio=None, max_delay=None):
        """
        Closes all shards for transfer and registration if there is not enough space
        """
        params = {
            'max_transfer_ratio': max_transfer_ratio,
            'max_registration_ratio': max_registration_ratio,
            'max_delay': max_delay,
        }
        return self.get('/close_for_load', params=params)

    def end_prepared_transaction(self, ttl_sc=None):
        """
        Terminates outdated prepared transactions that started more than 'ttl_sc' seconds ago for all shards
        """
        params = {
            'ttl_sc': ttl_sc
        }
        return self.get('/end_prepared_transaction', params=params)

    def init_pop3_folder(self):
        """
        Initializes user pop3 forlders for all shards
        """
        return self.get('/init_pop3_folder')

    def folder_archivation(self):
        """
        Archivates shared folders for all shards
        """
        return self.get('/folder_archivation')

    def purge_backups(self, ttl_days=None):
        """
        Purges user backups after 'ttl_days' days  for all shards
        """
        params = {
            'ttl_days': ttl_days,
        }
        return self.get('/purge_backups', params=params)

    def purge_chained_log(self):
        """
        Purges outdated chained log for all shards
        """
        return self.get('/purge_chained_log')

    def purge_deleted_box(self, ttl_days=None):
        """
        Purges deleted box after 'ttl_days' days  for all shards
        """
        params = {
            'ttl_days': ttl_days,
        }
        return self.get('/purge_deleted_box', params=params)

    def purge_synced_deleted_box(self):
        """
        Purges deleted box with synced attribute(for unsubscribed subscriptions).
        """
        return self.get('/purge_synced_deleted_box')

    def purge_storage(self, ttl_days=None, delete_chunk_size=None, jobs_count=None, job_no=None, max_delay=None):
        """
        Purges deleted messages from storage after 'ttl_days' days for all shards.
        Can be run in 'jobs_count' parallel jobs.
        """
        params = {
            'ttl_days': ttl_days,
            'jobs_count': jobs_count,
            'job_no': job_no,
            'delete_chunk_size': delete_chunk_size,
            'max_delay': max_delay,
        }
        return self.get('/purge_storage', params=params)

    def purge_deleted_user(self, ttl_days=None, force=None, jobs_count=None, job_no=None):
        """
        Purges deleted users from all shards after 'ttl_days' days.
        Purge user without check in sharddb if force == True.
        Can be run in 'jobs_count' parallel jobs.
        """
        params = {
            'ttl_days': ttl_days,
            'force': force,
            'jobs_count': jobs_count,
            'job_no': job_no,
        }
        return self.get('/purge_deleted_user', params=params)

    def purge_transferred_user(self, ttl_days=None, force=None, jobs_count=None, job_no=None):
        """
        Purges transfered users from all shards after 'ttl_days' days.
        Purge user without check in sharddb if force == True.
        Can be run in 'jobs_count' parallel jobs.
        """
        params = {
            'ttl_days': ttl_days,
            'force': force,
            'jobs_count': jobs_count,
            'job_no': job_no,
        }
        return self.get('/purge_transferred_user', params=params)

    def space_balancer(self, transfer_ratio=None, db_used_ratio=None, db_used_without_bloat_ratio=None,
                       load_type=None, shards_count=None, task_args={}):
        """
        Sets transfer tasks for users if there are not enough space in some shard
        Transfer to random 'shards_count' open shards with given 'load_type'.
        """
        params = {
            'transfer_ratio': transfer_ratio,
            'db_used_ratio': db_used_ratio,
            'db_used_without_bloat_ratio': db_used_without_bloat_ratio,
            'load_type': load_type,
            'shards_count': shards_count,
        }
        return self.post(
            '/space_balancer',
            params=params,
            headers={"Content-Type": "application/json"},
            json=task_args)

    def update_mailbox_size(self):
        """
        Update users mailbox database statistic for all shards
        """
        return self.get('/update_mailbox_size')

    def pg_partman_maintenance(self):
        """
        Runs pg_partman maintenance for all shards.
        It creates needed partitions and removes outdated partitions.
        """
        return self.get('/pg_partman_maintenance')

    def settings_export(self, from_uid=None, max_delay=None):
        """
        Export user settings into YT
        """
        params = {
            'from_uid': from_uid,
            'max_delay': max_delay,
        }
        return self.get('/settings_export', params=params)

    def reactivate_users(self, max_delay=None):
        """
        Reactivate users while freezing
        """
        params = {
            'max_delay': max_delay,
        }
        return self.get('/reactivate_users', params=params)

    def deactivate_users(self, max_delay=None, max_users_count=None):
        """
        Deactivate users while freezing
        """
        params = {
            'max_delay': max_delay,
            'max_users_count': max_users_count,
        }
        return self.get('/deactivate_users', params=params)

    def start_freezing_users(self, max_users_count=None):
        """
        Start freezing users(prepare to notify) while freezing
        """
        params = {
            'max_users_count': max_users_count,
        }
        return self.get('/start_freezing_users', params=params)

    def notify_users(self, max_delay=None, max_users_count=None):
        """
        Notify users while freezing
        """
        params = {
            'max_delay': max_delay,
            'max_users_count': max_users_count,
        }
        return self.get('/notify_users', params=params)

    def freeze_users(self, max_delay=None, max_users_count=None):
        """
        Freeze users while freezing
        """
        params = {
            'max_delay': max_delay,
            'max_users_count': max_users_count,
        }
        return self.get('/freeze_users', params=params)

    def archive_users(self, max_delay=None, max_users_count=None, messages_chunk_size=None):
        """
        Archive users while freezing
        """
        params = {
            'max_delay': max_delay,
            'max_users_count': max_users_count,
            'messages_chunk_size': messages_chunk_size,
        }
        return self.get('/archive_users', params=params)

    def purge_archives(self, max_delay=None):
        """
        Purge user outdated archives
        """
        params = {
            'max_delay': max_delay,
        }
        return self.get('/purge_archives', params=params)

    def clean_archives(self, max_delay=None):
        """
        Clean user archives
        """
        params = {
            'max_delay': max_delay,
        }
        return self.get('/clean_archives', params=params)

    def onetime_task(self, subtask=None, max_delay=None):
        """
        Onetime task
        """
        params = {
            'subtask': subtask,
            'max_delay': max_delay,
        }
        return self.get('/onetime_task', params=params)

    def pnl_estimation_export(self, from_uid=None, max_delay=None):
        """
        Export info about users for pnl estimation (message count, size in db, filter actions, etc) into YT
        """
        params = {
            'from_uid': from_uid,
            'max_delay': max_delay,
        }
        return self.get('/pnl_estimation_export', params=params)
