from .base import BaseClient


class ShardClient(BaseClient):
    def cleanup_doomed(self, shard_id=None):
        params = {
            'shard_id': shard_id
        }
        return self.get('/cleanup_doomed', params=params)

    def close_for_load(self, shard_id=None, max_transfer_ratio=None, max_registration_ratio=None, max_delay=None):
        """
        Closes shard for transfer and registration if there is not enough space in it.
        """
        params = {
            'shard_id': shard_id,
            'max_transfer_ratio': max_transfer_ratio,
            'max_registration_ratio': max_registration_ratio,
            'max_delay': max_delay,
        }
        return self.get('/close_for_load', params=params)

    def end_prepared_transaction(self, shard_id=None, ttl_sc=None):
        """
        Terminates outdated prepared transactions that started more than 'ttl_sc' seconds ago
        """
        params = {
            'shard_id': shard_id,
            'ttl_sc': ttl_sc
        }
        return self.get('/end_prepared_transaction', params=params)

    def init_pop3_folder(self, shard_id=None):
        """
        Initializes user pop3 forlders
        """
        params = {
            'shard_id': shard_id
        }
        return self.get('/init_pop3_folder', params=params)

    def folder_archivation(self, shard_id=None):
        """
        Archivates shared folders
        """
        params = {
            'shard_id': shard_id
        }
        return self.get('/folder_archivation', params=params)

    def purge_backups(self, shard_id=None, ttl_days=None):
        """
        Purges user backups after 'ttl_days' days.
        """
        params = {
            'shard_id': shard_id,
            'ttl_days': ttl_days,
        }
        return self.get('/purge_backups', params=params)

    def purge_chained_log(self, shard_id=None):
        """
        Purges outdated chained log
        """
        params = {
            'shard_id': shard_id
        }
        return self.get('/purge_chained_log', params=params)

    def purge_deleted_box(self, shard_id=None, ttl_days=None):
        """
        Purges deleted box after 'ttl_days' days.
        """
        params = {
            'shard_id': shard_id,
            'ttl_days': ttl_days,
        }
        return self.get('/purge_deleted_box', params=params)

    def purge_synced_deleted_box(self, shard_id=None):
        """
        Purges deleted box with synced attribute(for unsubscribed subscriptions).
        """
        params = {
            'shard_id': shard_id,
        }
        return self.get('/purge_synced_deleted_box', params=params)

    def purge_storage(self, shard_id=None, ttl_days=None, delete_chunk_size=None, jobs_count=None, job_no=None, max_delay=None):
        """
        Purges deleted messages from storage after 'ttl_days' days.
        Can be run in 'jobs_count' parallel jobs.
        """
        params = {
            'shard_id': shard_id,
            'ttl_days': ttl_days,
            'jobs_count': jobs_count,
            'job_no': job_no,
            'delete_chunk_size': delete_chunk_size,
            'max_delay': max_delay,
        }
        return self.get('/purge_storage', params=params)

    def purge_deleted_user(self, shard_id=None, ttl_days=None, force=None, jobs_count=None, job_no=None):
        """
        Purges deleted users from shard after 'ttl_days' days.
        Purge user without check in sharddb if force == True.
        Can be run in 'jobs_count' parallel jobs.
        """
        params = {
            'shard_id': shard_id,
            'ttl_days': ttl_days,
            'force': force,
            'jobs_count': jobs_count,
            'job_no': job_no,
        }
        return self.get('/purge_deleted_user', params=params)

    def purge_transferred_user(self, shard_id=None, ttl_days=None, force=None, jobs_count=None, job_no=None):
        """
        Purges transfered users from shard after 'ttl_days' days.
        Purge user without check in sharddb if force == True.
        Can be run in 'jobs_count' parallel jobs.
        """
        params = {
            'shard_id': shard_id,
            'ttl_days': ttl_days,
            'force': force,
            'jobs_count': jobs_count,
            'job_no': job_no,
        }
        return self.get('/purge_transferred_user', params=params)

    def space_balancer(self, shard_id=None, transfer_ratio=None, db_used_ratio=None, db_used_without_bloat_ratio=None,
                       load_type=None, shards_count=None, task_args={}):
        """
        Sets transfer tasks for users if there are not enough space in the shard.
        Transfer to random 'shards_count' open shards with given 'load_type'.
        """
        params = {
            'shard_id': shard_id,
            'transfer_ratio': transfer_ratio,
            'db_used_ratio': db_used_ratio,
            'db_used_without_bloat_ratio': db_used_without_bloat_ratio,
            'load_type': load_type,
            'shards_count': shards_count,
        }
        return self.post(
            '/space_balancer',
            params=params,
            headers={"Content-Type": "application/json"},
            json=task_args)

    def transfer_users(self, shard_id=None, messages_count=None, min_messages_per_user=None, max_messages_per_user=None,
                       load_type=None, shards_count=None, task_args={}):
        """
        Sets transfer tasks for users who have more than 'messages_count' letters in total.
        Transfer only users with messages count in specified intereval(min_messages_per_user, max_messages_per_user).
        Transfer to random 'shards_count' open shards with given 'load_type'.
        """
        params = {
            'shard_id': shard_id,
            'messages_count': messages_count,
            'min_messages_per_user': min_messages_per_user,
            'max_messages_per_user': max_messages_per_user,
            'load_type': load_type,
            'shards_count': shards_count,
        }
        return self.post(
            '/transfer_users',
            params=params,
            headers={"Content-Type": "application/json"},
            json=task_args)

    def transfer_active_users(self, shard_id=None, users_count=None, load_type=None, shards_count=None, task_args={}):
        """
        Sets transfer tasks for the most active users.
        Transfer to random 'shards_count' open shards with given 'load_type'.
        """
        params = {
            'shard_id': shard_id,
            'users_count': users_count,
            'load_type': load_type,
            'shards_count': shards_count,
        }
        return self.post(
            '/transfer_active_users',
            params=params,
            headers={"Content-Type": "application/json"},
            json=task_args)

    def update_mailbox_size(self, shard_id=None):
        """
        Update users mailbox database statistic
        """
        params = {
            'shard_id': shard_id
        }
        return self.get('/update_mailbox_size', params=params)

    def pg_partman_maintenance(self, shard_id=None):
        """
        Runs pg_partman maintenance.
        It creates needed partitions and removes outdated partitions.
        """
        params = {
            'shard_id': shard_id
        }
        return self.get('/pg_partman_maintenance', params=params)

    def settings_export(self, shard_id=None, from_uid=None, max_delay=None):
        """
        Export user settings into YT
        """
        params = {
            'shard_id': shard_id,
            'from_uid': from_uid,
            'max_delay': max_delay,
        }
        return self.get('/settings_export', params=params)

    def reactivate_users(self, shard_id=None, max_delay=None):
        """
        Reactivate users while freezing
        """
        params = {
            'shard_id': shard_id,
            'max_delay': max_delay,
        }
        return self.get('/reactivate_users', params=params)

    def deactivate_users(self, shard_id=None, max_delay=None, max_users_count=None):
        """
        Deactivate users while freezing
        """
        params = {
            'shard_id': shard_id,
            'max_delay': max_delay,
            'max_users_count': max_users_count,
        }
        return self.get('/deactivate_users', params=params)

    def start_freezing_users(self, shard_id=None, max_users_count=None):
        """
        Start freezing users(prepare to notify) while freezing
        """
        params = {
            'shard_id': shard_id,
            'max_users_count': max_users_count,
        }
        return self.get('/start_freezing_users', params=params)

    def notify_users(self, shard_id=None, max_delay=None, max_users_count=None):
        """
        Notify users while freezing
        """
        params = {
            'shard_id': shard_id,
            'max_delay': max_delay,
            'max_users_count': max_users_count,
        }
        return self.get('/notify_users', params=params)

    def freeze_users(self, shard_id=None, max_delay=None, max_users_count=None):
        """
        Freeze users while freezing
        """
        params = {
            'shard_id': shard_id,
            'max_delay': max_delay,
            'max_users_count': max_users_count,
        }
        return self.get('/freeze_users', params=params)

    def archive_users(self, shard_id=None, max_delay=None, max_users_count=None, messages_chunk_size=None):
        """
        Archive users while freezing
        """
        params = {
            'shard_id': shard_id,
            'max_delay': max_delay,
            'max_users_count': max_users_count,
            'messages_chunk_size': messages_chunk_size,
        }
        return self.get('/archive_users', params=params)

    def purge_archives(self, shard_id=None, max_delay=None):
        """
        Purge user outdated archives
        """
        params = {
            'shard_id': shard_id,
            'max_delay': max_delay,
        }
        return self.get('/purge_archives', params=params)

    def clean_archives(self, shard_id=None, max_delay=None):
        """
        Purge user outdated archives
        """
        params = {
            'shard_id': shard_id,
            'max_delay': max_delay,
        }
        return self.get('/clean_archives', params=params)

    def onetime_task(self, subtask=None, shard_id=None, max_delay=None):
        """
        Onetime task
        """
        params = {
            'subtask': subtask,
            'shard_id': shard_id,
            'max_delay': max_delay,
        }
        return self.get('/onetime_task', params=params)

    def pnl_estimation_export(self, shard_id=None, from_uid=None, max_delay=None):
        """
        Export info about users for pnl estimation (message count, size in db, filter actions, etc) into YT
        """
        params = {
            'shard_id': shard_id,
            'from_uid': from_uid,
            'max_delay': max_delay,
        }
        return self.get('/pnl_estimation_export', params=params)
