from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import WithPort, SubprocessComponent
from mail.devpack.lib.components.sharddb import ShardDb
from mail.devpack.lib.components.sharpei import Sharpei
from mail.devpack.lib.components.tvm_mulcagate import TvmMulcagate
from mail.devpack.lib.components.tvmtool import TvmTool
from mail.devpack.lib.components.fakebb import FakeBlackbox
from mail.devpack.lib.components.unimock import Unimock, UnimockApp
from mail.pg.huskydb.devpack.components.huskydb import HuskyDb
from mail.pg.queuedb.devpack.components.queuedb import QueueDb
from mail.callmeback.devpack.components.db import CallmebackDb
from mail.shiva.client.shiva import ShivaClient


class PassportMock(Unimock):
    NAME = "Passportmock"

    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(Unimock, cls).gen_config(port_generator, config=config)
        return dict(
            tvm={"client_id": next(port_generator)},
            **base
        )

    def __init__(self, env, components):
        self.tvm_id = env.config['unimock']['tvm']['client_id']
        super(Unimock, self).__init__(env, components)
        self.app = UnimockApp(self.logger, response_body='{"status":"ok"}')


class Shiva(SubprocessComponent, WithPort):
    NAME = "shiva"
    DEPS = [HuskyDb, QueueDb, CallmebackDb, ShardDb, Sharpei, TvmMulcagate, TvmTool, Unimock, PassportMock]

    def __init__(self, env, components):
        super(Shiva, self).__init__(env, components)
        self.tvm_client_id = self.config[self.NAME]['tvm']['client_id']
        self.__pyremock_port = self.config[self.NAME]['pyremock_port']
        self.bin_file = env.get_arcadia_bin('mail/shiva/bin/shiva')

    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(Shiva, cls).gen_config(port_generator, config=config)
        base.update({'tvm': {"client_id": next(port_generator)}})
        base.update({'pyremock_port': next(port_generator)})
        return base

    def init_root(self):
        helpers.mkdir_recursive(self.log_dir)

    def start(self):
        cmd = [
            self.bin_file,
            '--env', 'devpack',
            '--port', str(self.port),
        ]
        env = {
            'DEVPACK_HUSKYDB_PORT': str(self.components[HuskyDb].port()),
            'DEVPACK_QUEUEDB_PORT': str(self.components[QueueDb].port()),
            'DEVPACK_CALLMEBACKDB_PORT': str(self.components[CallmebackDb].port()),
            'DEVPACK_SHARDDB_PORT': str(self.components[ShardDb].port()),
            'DEVPACK_SHARPEI': self.components[Sharpei].api().location,
            'DEVPACK_MULCAGATE_HOST': self.components[TvmMulcagate].host,
            'DEVPACK_MULCAGATE_PORT': str(self.components[TvmMulcagate].port),
            'DEVPACK_CA_CERT': self.components[TvmMulcagate].ssl_cert_path,
            'DEVPACK_TVM_BB_ID': str(self.components[FakeBlackbox].tvm_id),
            'DEVPACK_BLACKBOX': self.components[FakeBlackbox].url,
            'DEVPACK_TVM_MDS_ID': str(self.components[TvmMulcagate].tvm_id),
            'DEVPACK_TVM_SHIVA_ID': str(self.tvm_client_id),
            'DEVPACK_TVMTOOL_URL': 'http://localhost:{port}'.format(port=self.components[TvmTool].port),
            'QLOUD_TVM_TOKEN': self.components[TvmTool].AUTHTOKEN,
            'DEVPACK_SHIVA': 'http://localhost:{port}'.format(port=self.port),
            'DEVPACK_SENDR': 'http://localhost:{port}'.format(port=self.components[Unimock].port),
            'DEVPACK_TVM_PASSPORT_ID': str(self.components[PassportMock].tvm_id),
            'DEVPACK_PASSPORT': 'http://localhost:{port}'.format(port=self.pyremock_port()),
            'DEVPACK_SURVEILLANCE': 'http://localhost:{port}'.format(port=self.pyremock_port()),
            'DEVPACK_S3': 'http://localhost:{port}'.format(port=self.pyremock_port()),
            'DEVPACK_TVM_S3_ID': str(self.components[Unimock].tvm_id),
        }
        with self.starting(cmd, env):
            helpers.wait_ping(self.logger, self.client().ping)

    def info(self):
        return {
            "root": self.root,
            "state": self.state,
            "bin_file": self.bin_file,
            "port": self.port,
        }

    def client(self):
        return ShivaClient(location="http://localhost:{}".format(self.port), logger=self.logger)

    def pyremock_port(self):
        return self.__pyremock_port
