import asyncio
import aiopg
import click
import uvloop
from aiohttp import web
from psycopg2.extras import RealDictCursor

from mail.python.theatre.stages.base import Stage
from mail.python.theatre.app.hostname import gethostname

from .props.admin_api import AdminApi
from .props.shard_api import ShardApi
from .props.maildb_api import MaildbApi
from .props.queuedb_api import QueuedbApi
from .props.huskydb_api import HuskydbApi
from .props.callmebackdb_api import CallmebackdbApi
from .props.util_api import UtilApi
from .props.shard.task import HuskydbEngine
from .props.http_middleware import middleware
from .roles.director import Director
from .settings.app import Settings


class ApiStage(Stage):
    def __init__(self, env: str, worker_name: str = None):
        self.worker_name = worker_name or gethostname()
        super().__init__(env=env, name='shiva-api', settings=Settings)

    async def create_app(self):
        huskydb_pool = await aiopg.create_pool(
            dsn=self.settings.huskydb.pg_dsn(username='sharpei'),
            minsize=1,
            maxsize=10,
            pool_recycle=600,
            cursor_factory=RealDictCursor,
        )
        huskydb = HuskydbEngine(huskydb_pool)
        director = Director(self.settings, huskydb, self.worker_name)
        self.app.update(director=director)
        await director.start()

        async def shutdown(_):
            await director.stop()

        self.app.on_shutdown.append(shutdown)
        self.setup_routes(
            admin_api=director.admin_api,
            shard_api=director.shard_api,
            maildb_api=director.maildb_api,
            huskydb_api=director.huskydb_api,
            queuedb_api=director.queuedb_api,
            callmebackdb_api=director.callmebackdb_api,
            util_api=director.util_api,
            director=director,
        )
        return self.app

    def setup_routes(
            self,
            admin_api: AdminApi,
            shard_api: ShardApi,
            maildb_api: MaildbApi,
            huskydb_api: HuskydbApi,
            queuedb_api: QueuedbApi,
            callmebackdb_api: CallmebackdbApi,
            util_api: UtilApi,
            director: Director,
            ):
        self.app.router.add_get('/ping', lambda _: web.Response(text='pong'))

        admin_app = web.Application(middlewares=[middleware])
        admin_api.bind_routes(admin_app)
        self.app.add_subapp('/admin', admin_app)

        shard_app = web.Application(middlewares=[middleware])
        shard_api.bind_routes(shard_app)
        self.app.add_subapp('/shard', shard_app)

        maildb_app = web.Application(middlewares=[middleware])
        maildb_api.bind_routes(maildb_app)
        self.app.add_subapp('/maildb', maildb_app)

        huskydb_app = web.Application(middlewares=[middleware])
        huskydb_api.bind_routes(huskydb_app)
        self.app.add_subapp('/huskydb', huskydb_app)

        queuedb_app = web.Application(middlewares=[middleware])
        queuedb_api.bind_routes(queuedb_app)
        self.app.add_subapp('/queuedb', queuedb_app)

        callmebackdb_app = web.Application(middlewares=[middleware])
        callmebackdb_api.bind_routes(callmebackdb_app)
        self.app.add_subapp('/callmebackdb', callmebackdb_app)

        util_app = web.Application(middlewares=[middleware])
        util_api.bind_routes(util_app)
        self.app.add_subapp('/util', util_app)

        self.app.router.add_get('/unistat', director.unistat_handler, name='unistat')


@click.command()
@click.option('--host', default='::0', help='IP addr to bind to')
@click.option('--port', default=80, help='HTTP port to listen')
@click.option('--env', help='Environment name')
@click.option('--worker-name', help='Worker name to be used for task acquisition')
def run_main(host, port, env=None, worker_name=None):
    asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())

    stage = ApiStage(env=env, worker_name=worker_name)
    stage.run(host=host, port=port)


if __name__ == '__main__':
    run_main()
