def read_shards(rows):
    return {str(row['shard_id']): row for row in rows}


async def get_shards(huskydb, shard_id, cluster_id, shard_type):
    async with huskydb.connection() as conn:
        async with conn.cursor() as cur:
            await cur.execute(
                '''SELECT *
                     FROM shiva.shards
                    WHERE (%(shard_id)s is NULL or shard_id = %(shard_id)s)
                      AND (%(cluster_id)s is NULL or cluster_id = %(cluster_id)s)
                      AND (%(shard_type)s is NULL or shard_type = %(shard_type)s)
                ''',
                dict(
                    shard_id=shard_id,
                    cluster_id=cluster_id,
                    shard_type=shard_type
                )
            )
            return read_shards([row async for row in cur])


async def delete_shard(huskydb, shard_id):
    async with huskydb.connection() as conn:
        async with conn.cursor() as cur:
            await cur.execute(
                '''DELETE FROM shiva.shards
                    WHERE shard_id = %(shard_id)s
                RETURNING *
                ''',
                dict(shard_id=shard_id)
            )
            return read_shards([row async for row in cur])


async def add_shard(huskydb, shard_id, cluster_id, shard_name, shard_type, load_type, can_transfer_to, migration, priority, disk_size):
    async with huskydb.connection() as conn:
        async with conn.cursor() as cur:
            await cur.execute(
                '''INSERT INTO shiva.shards (shard_id, cluster_id, shard_type, load_type, can_transfer_to, migration, priority, disk_size, shard_name)
                   VALUES (%(shard_id)s, %(cluster_id)s, %(shard_type)s, %(load_type)s, %(can_transfer_to)s, %(migration)s, %(priority)s, %(disk_size)s, %(shard_name)s)
                RETURNING *
                ''',
                dict(
                    shard_id=shard_id,
                    cluster_id=cluster_id,
                    shard_type=shard_type,
                    load_type=load_type,
                    can_transfer_to=can_transfer_to,
                    migration=migration,
                    priority=priority,
                    disk_size=disk_size,
                    shard_name=shard_name,
                )
            )
            return read_shards([row async for row in cur])


async def update_shard(huskydb, shard_id, cluster_id, shard_name, shard_type, load_type, can_transfer_to, migration, priority, disk_size):
    async with huskydb.connection() as conn:
        async with conn.cursor() as cur:
            await cur.execute(
                '''UPDATE shiva.shards
                      SET cluster_id = COALESCE(%(cluster_id)s, cluster_id),
                          shard_type = COALESCE(%(shard_type)s, shard_type),
                          load_type = COALESCE(%(load_type)s, load_type),
                          can_transfer_to = COALESCE(%(can_transfer_to)s, can_transfer_to),
                          migration = COALESCE(%(migration)s, migration),
                          priority = COALESCE(%(priority)s, priority),
                          disk_size = COALESCE(%(disk_size)s, disk_size),
                          shard_name = COALESCE(%(shard_name)s, shard_name)
                    WHERE shard_id = %(shard_id)s
                RETURNING *
                ''',
                dict(
                    shard_id=shard_id,
                    cluster_id=cluster_id,
                    shard_name=shard_name,
                    shard_type=shard_type,
                    load_type=load_type,
                    can_transfer_to=can_transfer_to,
                    migration=migration,
                    priority=priority,
                    disk_size=disk_size,
                )
            )
            return read_shards([row async for row in cur])
