from asyncio import shield
from aiohttp import web

from mail.shiva.stages.api.roles.shard_worker import ShardWorker, Task
from mail.shiva.stages.api.settings.app import Settings

from .shard.task import plan_task, get_task, HuskydbEngine
from .errors import MissingArgumentError


class BaseApi:
    def __init__(self, settings: Settings, huskydb: HuskydbEngine):
        self._settings = settings
        self._huskydb = huskydb

    def get_required_parameter(self, request: web.Request, param_name: str):
        param = request.query.get(param_name, None)
        if param is None:
            raise MissingArgumentError(f'missing required argument: "{param_name}"')
        return param


class WorkerApi(BaseApi):
    def __init__(self, settings: Settings, huskydb: HuskydbEngine, shard_worker: ShardWorker):
        super().__init__(settings, huskydb)
        self._shard_worker = shard_worker

    async def run_task_cancellable(self, params, job):
        task = await plan_task(
            huskydb=self._huskydb,
            worker_name=self._shard_worker.worker_name,
            params=params)
        if not task:
            task = await get_task(huskydb=self._huskydb, params=params)
            if not task:
                task = {}
            task['status'] = 'already_running'
        else:
            task['status'] = 'planned'
            self._shard_worker.put_nowait(Task(job=job, params=params))
        return task

    async def run_task(self, params, job):
        return await shield(self.run_task_cancellable(params, job))
