import logging
import aiopg
from dataclasses import dataclass
from datetime import timedelta

from mail.shiva.stages.api.settings.callmebackdb import CallmebackDbSettings
from mail.shiva.stages.api.props.shard.task import TaskParams

log = logging.getLogger(__name__)
CLEAN_TIMEOUT = 600


async def clean_events(conn, clean_ttl, chunk_size):
    while True:
        async with conn.cursor(timeout=CLEAN_TIMEOUT) as cur:
            await cur.execute(
                '''
                DELETE
                  FROM reminders.events
                 WHERE event_id IN (
                    SELECT event_id FROM reminders.events
                     WHERE run_at < now() - %(clean_ttl)s
                       AND status != 'pending'
                     LIMIT %(chunk_size)s)
                ''',
                dict(
                    clean_ttl=clean_ttl,
                    chunk_size=chunk_size,
                )
            )
            log.info('clean_events: successfully purged %d records', cur.rowcount)
            if cur.rowcount < chunk_size:
                return


async def clean_change_log(conn, clean_ttl, chunk_size):
    while True:
        async with conn.cursor(timeout=CLEAN_TIMEOUT) as cur:
            await cur.execute(
                '''
                DELETE
                  FROM reminders.change_log
                 WHERE cid IN (
                     SELECT cid FROM reminders.change_log
                      WHERE at < now() - %(clean_ttl)s
                      LIMIT %(chunk_size)s)
                ''',
                dict(
                    clean_ttl=clean_ttl,
                    chunk_size=chunk_size,
                )
            )
            log.info('clean_change_log: successfully purged %d records', cur.rowcount)
            if cur.rowcount < chunk_size:
                return


@dataclass
class CleanParams(TaskParams):
    task_name: str = 'callmebackdb_clean'
    shard_id: int = 0
    callmebackdb: CallmebackDbSettings = None
    clean_ttl: timedelta = timedelta(days=30)
    chunk_size: int = 50000


async def clean(params: CleanParams, stats):
    async with aiopg.connect(params.callmebackdb.pg_dsn()) as conn:
        await clean_events(conn, params.clean_ttl, params.chunk_size)
        await clean_change_log(conn, params.clean_ttl, params.chunk_size)
