from typing import Dict
import logging
import ujson
from aiohttp import web, web_exceptions

from .errors import BadRequest

log = logging.getLogger(__name__)


HTTP_200 = 200
HTTP_400 = 400
HTTP_500 = 500
HTTP_501 = 501


def dumps(data):
    return ujson.dumps(data, indent=4)


def success(data: Dict) -> web.Response:
    return web.json_response(data, status=HTTP_200, dumps=dumps)


def error(code: int, ex: Exception) -> web.Response:
    data = {'error': {'reason': ex.__class__.__name__, 'message': str(ex)}}
    return web.json_response(data, status=code, dumps=dumps)


@web.middleware
async def middleware(request, handler):
    try:
        return success(await handler(request))
    except BadRequest as ex:
        log.error('Request failed due to error <%s>', str(ex))
        return error(HTTP_400, ex)
    except NotImplementedError as ex:
        return error(HTTP_501, ex)
    except web_exceptions.HTTPException:
        raise
    except Exception as ex:
        log.exception(ex)
        return error(HTTP_500, ex)
