import logging
import asyncio
import random
from aiohttp import web

from mail.python.theatre.profiling.http import ProfiledClientSession
from mail.shiva.stages.api.settings.log import http_logger
from mail.shiva.stages.api.settings.app import Settings
from mail.python.theatre.profiling.typing import Metrics
from .base_api import BaseApi
from .shard.task import HuskydbEngine
from .stats import Stats

log = logging.getLogger(__name__)


class MaildbApi(BaseApi):
    def __init__(self, settings: Settings, huskydb: HuskydbEngine):
        super().__init__(settings, huskydb)
        self.stats = Stats()

    def bind_routes(self, app: web.Application):
        app.router.add_get('/cleanup_doomed', self.cleanup_doomed)
        app.router.add_get('/close_for_load', self.close_for_load)
        app.router.add_get('/end_prepared_transaction', self.end_prepared_transaction)
        app.router.add_get('/init_pop3_folder', self.init_pop3_folder)
        app.router.add_get('/folder_archivation', self.folder_archivation)
        app.router.add_get('/pg_partman_maintenance', self.pg_partman_maintenance)
        app.router.add_get('/purge_backups', self.purge_backups)
        app.router.add_get('/purge_chained_log', self.purge_chained_log)
        app.router.add_get('/purge_deleted_box', self.purge_deleted_box)
        app.router.add_get('/purge_synced_deleted_box', self.purge_synced_deleted_box)
        app.router.add_get('/purge_storage', self.purge_storage)
        app.router.add_get('/purge_deleted_user', self.purge_deleted_user)
        app.router.add_get('/purge_transferred_user', self.purge_transferred_user)
        app.router.add_post('/space_balancer', self.space_balancer)
        app.router.add_get('/update_mailbox_size', self.update_mailbox_size)
        app.router.add_get('/settings_export', self.settings_export)
        app.router.add_get('/pnl_estimation_export', self.pnl_estimation_export)

        # jobs for freezing users
        app.router.add_get('/reactivate_users', self.reactivate_users)
        app.router.add_get('/deactivate_users', self.deactivate_users)
        app.router.add_get('/start_freezing_users', self.start_freezing_users)
        app.router.add_get('/notify_users', self.notify_users)
        app.router.add_get('/freeze_users', self.freeze_users)
        app.router.add_get('/archive_users', self.archive_users)
        app.router.add_get('/purge_archives', self.purge_archives)
        app.router.add_get('/clean_archives', self.clean_archives)

        app.router.add_get('/onetime_task', self.onetime_task)

    async def shard_request(self, http_session, method, uri, **kwargs):
        sleep_before_start = random.random() * self._settings.shiva.max_delay
        await asyncio.sleep(sleep_before_start)

        for _ in range(self._settings.shiva.tries):
            try:
                async with http_session.request(
                        method=method,
                        url=f'{self._settings.shiva.location}/shard/{uri}',
                        **kwargs,
                ) as response:
                    if response.status != 200:
                        reason = (await response.text()).strip()
                        log.warning(f'non-200 shiva code, gonna retry: {response.status}, reason: {reason}')
                    else:
                        return True
            except Exception as exp:
                log.warning(f'got exception during shiva shard request, gonna retry: {exp}')
            await asyncio.sleep(self._settings.shiva.try_sleep)

        log.error('cannot get valid shiva shard response: try limit')
        return False

    async def get_shards(self):
        async with self._huskydb.connection() as conn:
            async with conn.cursor() as cur:
                await cur.execute(
                    '''
                    SELECT shard_id
                      FROM shiva.shards
                    '''
                )
                return [rec['shard_id'] async for rec in cur]

    async def maildb_requests(self, method, uri, params, **kwargs):
        shards = await self.get_shards()
        async with ProfiledClientSession(metrics=self.stats, logger=http_logger.get_logger(),
                                         timeout=self._settings.shiva.timeout) as http_session:
            tasks = [asyncio.create_task(self.shard_request(
                http_session=http_session,
                method=method,
                uri=uri,
                params={**params, 'shard_id': shard_id},
                **kwargs
            )) for shard_id in shards]
            results = await asyncio.gather(*tasks)
            return {
                "shards_request_count": len(results),
                "successful_count": results.count(True),
            }

    async def cleanup_doomed(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='cleanup_doomed',
            params=request.query)

    async def close_for_load(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='close_for_load',
            params=request.query)

    async def end_prepared_transaction(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='end_prepared_transaction',
            params=request.query)

    async def init_pop3_folder(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='init_pop3_folder',
            params=request.query)

    async def folder_archivation(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='folder_archivation',
            params=request.query)

    async def pg_partman_maintenance(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='pg_partman_maintenance',
            params=request.query)

    async def purge_backups(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='purge_backups',
            params=request.query)

    async def purge_chained_log(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='purge_chained_log',
            params=request.query)

    async def purge_deleted_box(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='purge_deleted_box',
            params=request.query)

    async def purge_synced_deleted_box(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='purge_synced_deleted_box',
            params=request.query)

    async def purge_storage(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='purge_storage',
            params=request.query)

    async def purge_deleted_user(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='purge_deleted_user',
            params=request.query)

    async def purge_transferred_user(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='purge_transferred_user',
            params=request.query)

    async def space_balancer(self, request: web.Request):
        data = await request.text()
        return await self.maildb_requests(
            method='post',
            uri='space_balancer',
            params=request.query,
            headers={"Content-Type": "application/json"},
            data=data)

    async def update_mailbox_size(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='update_mailbox_size',
            params=request.query)

    async def settings_export(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='settings_export',
            params=request.query)

    async def reactivate_users(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='reactivate_users',
            params=request.query)

    async def deactivate_users(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='deactivate_users',
            params=request.query)

    async def start_freezing_users(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='start_freezing_users',
            params=request.query)

    async def notify_users(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='notify_users',
            params=request.query)

    async def freeze_users(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='freeze_users',
            params=request.query)

    async def archive_users(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='archive_users',
            params=request.query)

    async def purge_archives(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='purge_archives',
            params=request.query)

    async def clean_archives(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='clean_archives',
            params=request.query)

    async def pnl_estimation_export(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='pnl_estimation_export',
            params=request.query)

    async def onetime_task(self, request: web.Request):
        return await self.maildb_requests(
            method='get',
            uri='onetime_task',
            params=request.query)

    def metrics(self) -> Metrics:
        return self.stats.get()
