import asyncio
import json
from aiohttp.client_exceptions import ClientConnectorError
from yarl import URL

from mail.shiva.stages.api.props.logger import get_uid_logger

log = get_uid_logger(__name__)


class Metrics(object):
    def __init__(self, stats, uid):
        self.logger = log
        self._stats = stats
        self.uid = uid

    def passport_update_user_state_retriable_error(self, msg):
        self.increase('passport_update_user_state_retriable_error')
        self.logger.error(msg, uid=self.uid)

    def passport_exception(self, msg):
        self.increase('passport_exception')
        self.logger.error(msg, uid=self.uid)

    def increase(self, name):
        self._stats.increase_task_meter(f"{name}")


async def passport_update_user_state(client, cfg, uid, passport_tvm_ticket, stats):
    params = {'consumer': 'shiva'}
    headers = {'x-ya-service-ticket': str(passport_tvm_ticket)}
    body = {'uid': uid, 'is_mailbox_frozen': 'yes'}

    retriable_errors = {
        "backend.blackbox_failed",
        "backend.yasms_failed",
        "backend.database_failed",
        "backend.redis_failed",
    }

    metrics = Metrics(stats, uid)

    for i in range(cfg.tries):
        try:
            async with client.post(url=URL(cfg.location + '/2/account/options/'), params=params, headers=headers,
                                   data=body) as r:
                if r.status // 100 == 5:
                    metrics.passport_update_user_state_retriable_error(f"Unexpected response status from passport: {r.status}")
                    continue
                elif r.status // 100 != 2:
                    metrics.increase('passport_update_user_state_fail')
                    raise RuntimeError(f"Can't update user state in passport: not retriable http code: {r.status}")
                else:
                    resp = await r.text()
                    js = json.loads(resp)
                    if not isinstance(js, dict):
                        metrics.passport_update_user_state_retriable_error(f"Unexpected response from passport: {js}")
                        continue
                    if 'errors' not in js:
                        return
                    elif set(js['errors']).issubset(retriable_errors):
                        metrics.passport_update_user_state_retriable_error(f"Got retriable error from passport: {js['errors']}")
                        continue
                    else:
                        metrics.increase('passport_update_user_state_fail')
                        raise RuntimeError(f"Can't update user state in passport: not retriable error {js['errors']}")
        except (asyncio.TimeoutError, ClientConnectorError) as exc:
            metrics.passport_exception(f'Got exception during passport request: {exc}')
    metrics.increase('passport_update_user_state_fail')
    raise RuntimeError("Can't update user state in passport: retries ended")
