import json
import logging
import asyncio

from mail.shiva.stages.api.settings.sharpei import SharpeiSettings
from mail.python.theatre.profiling.http import ProfiledClientSession
from mail.shiva.stages.api.settings.log import http_logger

log = logging.getLogger(__name__)


class SharpeiError(RuntimeError):
    pass


def make_dsn(shard, user):
    hosts = []
    ports = []
    dbname = ''
    for shard_instance in shard:
        hosts.append(shard_instance['address']['host'])
        ports.append(str(shard_instance['address']['port']))
        dbname = shard_instance['address']['dbname']
    host = ','.join(hosts)
    port = ','.join(ports)
    return f'host={host} port={port} dbname={dbname} user={user} target_session_attrs=read-write'


async def get_shard_dsn(cfg: SharpeiSettings, user: str, shard_id, stats):
    async with ProfiledClientSession(metrics=stats, logger=http_logger.get_logger(), timeout=cfg.timeout) as client:
        for _ in range(cfg.tries):
            try:
                async with client.get(url=f'{cfg.location}/stat') as resp:
                    if resp.status != 200:
                        log.warning(f'strange sharpei response status, gonna retry: {resp.status}')
                    else:
                        stat_resp = json.loads(await resp.text())
                        if str(shard_id) not in stat_resp:
                            raise SharpeiError(f'shard_id={shard_id} not found in /stat response: {stat_resp}')
                        return make_dsn(stat_resp[str(shard_id)], user)
            except SharpeiError:
                raise
            except Exception as exp:
                log.warning(f'got exception during sharpei request, gonna retry: {exp}')
            await asyncio.sleep(cfg.try_sleep)

    raise SharpeiError('cannot get valid sharpei response: try limit')


async def get_shard_by_uid(cfg: SharpeiSettings, uid, stats):
    async with ProfiledClientSession(metrics=stats, logger=http_logger.get_logger(), timeout=cfg.timeout) as client:
        for _ in range(cfg.tries):
            try:
                async with client.get(url=f'{cfg.location}/conninfo', params={
                    'uid': str(uid),
                    'mode': 'all',
                    'force': 'true',
                }) as resp:
                    if resp.status != 200:
                        log.warning(f'strange sharpei response status, gonna retry: {resp.status}')
                    else:
                        return json.loads(await resp.text())
            except SharpeiError:
                raise
            except Exception as exp:
                log.warning(f'got exception during sharpei request, gonna retry: {exp}')
            await asyncio.sleep(cfg.try_sleep)

    raise SharpeiError('cannot get valid sharpei response: try limit')


async def get_shard_id_by_uid(cfg: SharpeiSettings, uid, stats):
    resp = await get_shard_by_uid(cfg, uid, stats)
    try:
        return int(resp['id'])
    except KeyError:
        raise SharpeiError(f'shard_id not found in /conninfo response: {resp}')
    except ValueError:
        raise SharpeiError(f'bad shard_id in /conninfo response: {resp}')


async def get_shard_dsn_by_uid(cfg: SharpeiSettings, user: str, uid, stats):
    resp = await get_shard_by_uid(cfg, uid, stats)
    try:
        return make_dsn(resp['databases'], user)
    except KeyError:
        raise SharpeiError(f'shard_id not found in /conninfo response: {resp}')


async def get_sharddb_dsn(cfg: SharpeiSettings, user: str, stats):
    async with ProfiledClientSession(metrics=stats, logger=http_logger.get_logger(), timeout=cfg.timeout) as client:
        for _ in range(cfg.tries):
            try:
                async with client.get(url=f'{cfg.location}/sharddb_stat') as resp:
                    if resp.status != 200:
                        log.warning(f'strange sharpei response status, gonna retry: {resp.status}')
                    else:
                        stat_resp = json.loads(await resp.text())
                        addr = next((host['address'] for host in stat_resp if host['role'] == 'master'), None)
                        if not addr:
                            raise SharpeiError(f'master not found in /sharddb_stat response: {stat_resp}')
                        addr['user'] = user
                        return 'host={host} port={port} dbname={dbname} user={user}'.format(**addr)
            except SharpeiError:
                raise
            except Exception as exp:
                log.warning(f'got exception during sharpei request, gonna retry: {exp}')
            await asyncio.sleep(cfg.try_sleep)
    raise SharpeiError('cannot get valid sharpei response: try limit')
