import logging
import asyncio

from mail.shiva.stages.api.settings.surveillance import SurveillanceSettings
from mail.python.theatre.profiling.http import ProfiledClientSession
from mail.shiva.stages.api.settings.log import http_logger

log = logging.getLogger(__name__)


class SurveillanceError(RuntimeError):
    pass


def parse_response(resp_text):
    users = set()
    for line in resp_text.splitlines():
        line = line.strip()
        if not line:
            continue
        user_pair = line.split(';')
        if len(user_pair) != 2:
            raise SurveillanceError(f'got strange line in response from surveillance: {line}')
        users.add(user_pair[1])
    return users


async def get_surveillance_users(cfg: SurveillanceSettings, stats):
    if cfg.location is None:
        return set()
    async with ProfiledClientSession(metrics=stats, logger=http_logger.get_logger(), timeout=cfg.timeout) as client:
        for _ in range(cfg.tries):
            try:
                async with client.get(url=f'{cfg.location}/monitoring/?uids=1') as resp:
                    if resp.status != 200:
                        log.warning(f'strange response from surveillance, gonna retry: {resp.status}')
                    else:
                        return parse_response(await resp.text())
            except SurveillanceError:
                raise
            except Exception as exp:
                log.warning(f'got exception during surveillance request, gonna retry: {exp}')
            await asyncio.sleep(cfg.try_sleep)

    raise SurveillanceError('cannot get valid surveillance response: try limit')
