import asyncio
import tenacity

from .export_helper import randomize_start_time
from .cursor_provider import create_cursor_provider
from mail.shiva.stages.api.props.logger import get_uid_logger
from mail.shiva.stages.api.props.services.archive import ArchiveStorage
from mail.shiva.stages.api.props.services.sharpei import get_shard_id_by_uid
from mail.shiva.stages.api.props.shard.clean_archives import clean_stids, get_stids_to_clean

log = get_uid_logger(__name__)

SLEEP_ON_PG_FAILS = 120


async def get_next_user_for_clean(conn, uid, jobs_count, job_no):
    while True:
        try:
            async with conn.cursor() as cur:
                await cur.execute(
                    '''
                        SELECT uid
                          FROM mail.users mu JOIN mail.archives ma USING (uid)
                         WHERE is_here
                           AND mu.state = 'archived'
                           AND ma.state  = 'archivation_complete'
                           AND ma.message_count > 0
                           AND uid %% %(jobs_count)s = %(job_no)s
                           AND uid > %(uid)s
                         ORDER BY uid
                         LIMIT 1
                    ''',
                    dict(
                        uid=uid,
                        jobs_count=jobs_count,
                        job_no=job_no,
                    )
                )
                res = await cur.fetchone()
                return res and res['uid']
        except Exception as exc:
            log.exception(f'Got exception: {exc}', uid=uid)
            await asyncio.sleep(SLEEP_ON_PG_FAILS)


async def update_archive(conn, uid):
    async with conn.cursor() as cur:
        await cur.execute(
            '''
            update mail.archives
            set message_count = 0
            WHERE uid = %(uid)s
            ''',
            dict(
                uid=uid,
            )
        )


async def clean_user_archive(conn, archive_storage, uid):
    cleaned_msgs_count = 0
    try:
        user_keys = await archive_storage.list_user_objects(uid)
        for key in user_keys:
            messages = await archive_storage.get_messages(key)
            stids_to_clean = get_stids_to_clean(messages)
            await clean_stids(conn, uid, stids_to_clean)
            await archive_storage.delete_objects([key])
            cleaned_msgs_count += len(stids_to_clean)

        await update_archive(conn, uid)
        log.info('successfully clean existing user archive', uid=uid)
        return cleaned_msgs_count
    except Exception as exc:
        log.exception(f'Got exception: {exc}', uid=uid)
        return cleaned_msgs_count


@tenacity.retry(reraise=True, wait=tenacity.wait_fixed(1), stop=tenacity.stop_after_attempt(5))
async def refresh_tvm(archive_storage):
    await archive_storage.refresh_ticket()


async def clean_existing_archives(params, stats):
    await randomize_start_time(max_delay=params.max_delay)

    archive_storage = ArchiveStorage(
        s3api_settings=params.settings.s3api,
        s3_id=params.settings.tvm.s3_id,
        tvm=params.tvm_tickets,
        stats=stats,
        auto_refresh_tvm_ticket=False,
    )

    async with create_cursor_provider(params, stats) as conn:
        uid = 0
        processed_uids = 0
        cleaned_msgs_count = 0
        while True:
            if processed_uids % 1000 == 0:
                await refresh_tvm(archive_storage)
            processed_uids += 1
            uid = await get_next_user_for_clean(
                conn=conn,
                uid=uid,
                jobs_count=params.jobs_count,
                job_no=params.job_no,
            )
            if not uid:
                return
            if params.shard_id != await get_shard_id_by_uid(params.sharpei, uid, stats):
                log.info('skipping user archive, user in different shard', uid=uid)
                continue
            cleaned_msgs_count += await clean_user_archive(conn, archive_storage, uid)
            if cleaned_msgs_count >= (params.max_count / params.jobs_count):
                return
