import aiopg
import psycopg2
from psycopg2.extras import RealDictCursor
from dataclasses import dataclass
from datetime import timedelta
from typing import Dict
from frozendict import frozendict

from .helpers import chunks
from mail.shiva.stages.api.props.services.sharpei import get_shard_dsn
from .task import TaskParams
from mail.shiva.stages.api.props.logger import get_uid_logger

log = get_uid_logger(__name__)
PURGE_TIMEOUT = 300
GET_DOOMED_FOLDERS_TIMEOUT = 600


@dataclass
class DoomedFolder:
    uid: str = None
    fid: str = None
    type: str = None


async def get_doomed_folders(conn, doomed_folder_types, chunk_size):
    last_uid = 0
    while True:
        async with conn.cursor(timeout=GET_DOOMED_FOLDERS_TIMEOUT) as cur:
            await cur.execute(
                '''
                SELECT uid, fid, type
                  FROM mail.folders f
                 WHERE EXISTS (
                  SELECT 1
                    FROM mail.users u
                   WHERE u.uid = f.uid
                     AND u.is_here)
                   AND message_count > 0
                   AND type = ANY(%(doomed_folder_types)s::mail.folder_types[])
                   AND NOT EXISTS (SELECT 1 FROM mailish.accounts a WHERE a.uid = f.uid)
                   AND uid > %(last_uid)s
                   AND (
                        (type = 'hidden_trash' AND NOT EXISTS (
                            SELECT 1
                              FROM settings.settings s
                             WHERE s.uid = f.uid
                               AND value->'single_settings'->'mail_b2b_admin_search_allowed' = '"on"'
                               AND value->'single_settings'->'mail_b2b_admin_search_enabled' = '"on"'
                            )
                        ) OR type != 'hidden_trash'
                   )
                   ORDER BY uid
                   LIMIT %(chunk_size)s
                ''',
                dict(
                    doomed_folder_types=doomed_folder_types,
                    last_uid=last_uid,
                    chunk_size=chunk_size,
                )
            )
            chunk = [DoomedFolder(**r) async for r in cur]

            if chunk:
                last_uid = chunk[-1].uid
                yield chunk

            if len(chunk) < chunk_size:
                return


async def get_doomed_mids(conn, folder, folder_ttl, chunk_size):
    while True:
        async with conn.cursor() as cur:
            await cur.execute(
                '''
                SELECT mid
                  FROM mail.box mb
                 WHERE doom_date IS NOT NULL
                   AND fid = %(fid)s
                   AND uid = %(uid)s
                   AND doom_date < now() - %(max_days)s
                 LIMIT %(chunk_size)s
                ''',
                dict(
                    uid=folder.uid,
                    fid=folder.fid,
                    max_days=folder_ttl,
                    chunk_size=chunk_size,
                )
            )
            doomed_mids = [row['mid'] async for row in cur]
            yield doomed_mids
            if len(doomed_mids) < chunk_size:
                break


async def delete_mids(conn, uid, mids):
    try:
        async with conn.cursor(timeout=PURGE_TIMEOUT) as cur:
            await cur.execute(
                'SELECT * FROM code.delete_messages(%(uid)s, %(mids)s)',
                dict(
                    uid=uid,
                    mids=mids
                )
            )
    except psycopg2.DatabaseError:
        log.exception('exception while deleting old messages', uid=uid)
    else:
        log.info(f'successfully deleted {len(mids)} old messages', uid=uid)


@dataclass
class СleanupDoomedParams(TaskParams):
    task_name: str = 'cleanup_doomed'
    folders_ttl: Dict[str, timedelta] = frozendict(
        spam=timedelta(days=10),
        trash=timedelta(days=30),
        hidden_trash=timedelta(days=185)
    )
    get_doomed_folders_chunk_size: int = 1000
    get_doomed_mids_chunk_size: int = 10000
    chunk_to_delete: int = 100


async def shard_cleanup_doomed(params: СleanupDoomedParams, stats):
    async with aiopg.connect(await get_shard_dsn(params.sharpei, params.db_user, params.shard_id, stats), cursor_factory=RealDictCursor) as conn:
        async for doomed_folder_chunk in get_doomed_folders(
            conn,
            list(params.folders_ttl),
            params.get_doomed_folders_chunk_size,
        ):
            for doomed_folder in doomed_folder_chunk:
                try:
                    async for doomed_mids in get_doomed_mids(
                        conn,
                        doomed_folder,
                        params.folders_ttl[doomed_folder.type],
                        params.get_doomed_mids_chunk_size,
                    ):
                        for mids in chunks(doomed_mids, params.chunk_to_delete):
                            await delete_mids(
                                conn,
                                doomed_folder.uid,
                                list(mids)
                            )
                except Exception as exc:
                    log.exception(f'Got exception while purging fid={doomed_folder.fid}: {exc}', uid=doomed_folder.uid)
