import tenacity
from dataclasses import dataclass
from datetime import timedelta, datetime

import yt.wrapper as yt
import yt.yson as yson
from yql.api.v1.client import YqlClient

from .helpers import run_in_executor, db_general_retry
from .task import TaskParams
from .freeze_helper import db_update_user_state, db_get_user_state, is_live_user
from .export_helper import randomize_start_time
from .cursor_provider import create_cursor_provider, locked_transactional_cursor
from mail.shiva.stages.api.props.logger import get_uid_logger

log = get_uid_logger(__name__)
GET_USERS_TIMEOUT = 1000
YQL_RETRY_DELAY = 300


@db_general_retry
async def get_users_chunk(conn, state_ttl, last_uid, chunk_size):
    async with conn.cursor(timeout=GET_USERS_TIMEOUT) as cur:
        await cur.execute(
            '''
                SELECT uid
                  FROM mail.users u
                 WHERE is_here
                   AND NOT is_deleted
                   AND state = 'active'
                   AND last_state_update < (now() - %(state_ttl)s)
                   AND uid > %(last_uid)s
                   AND uid NOT BETWEEN 1130000000000000 AND 1139999999999999
                   AND NOT EXISTS (SELECT 1 FROM mailish.accounts a WHERE a.uid = u.uid)
                   AND NOT EXISTS (
                       SELECT 1
                         FROM filters.actions a
                         JOIN filters.rules r
                        USING (uid, rule_id)
                        WHERE a.uid = u.uid
                          AND enabled
                          AND oper IN ('forward', 'forwardwithstore', 'reply', 'notify'))
                   AND NOT EXISTS (
                       SELECT 1
                         FROM settings.settings s
                        WHERE s.uid = u.uid
                          AND (value->'single_settings'->'has_pro_interface' = '"on"'
                           OR value->'single_settings'->'is_ad_disabled_via_billing' = '"on"'))
                 ORDER BY uid
                 LIMIT %(chunk_size)s
            ''',
            dict(
                state_ttl=state_ttl,
                last_uid=last_uid,
                chunk_size=chunk_size,
            )
        )
        return [rec['uid'] async for rec in cur]


async def get_users(conn, state_ttl, chunk_size):
    last_uid = 0
    while True:
        chunk = await get_users_chunk(conn, state_ttl, last_uid, chunk_size)

        if chunk:
            last_uid = chunk[-1]
            yield chunk

        if len(chunk) < chunk_size:
            return


async def deactivate_user(conn, uid):
    try:
        async with locked_transactional_cursor(conn, uid) as cur:
            user_state = await db_get_user_state(cur, uid)
            if not is_live_user(user_state) or user_state.state != 'active':
                log.info('user was skipped, because his state was changed', uid=uid)
                return
            await db_update_user_state(cur, user_state, 'inactive')
    except Exception as exc:
        log.exception(f'Got exception: {exc}', uid=uid)
        return
    log.info('user was successfully deactivated', uid=uid)


async def write_users_to_yt(yt_client, table, data):
    def _proc(yt_client, table, data):
        yt_client.write_table(
            table=yt.TablePath(table, append=True, sorted_by=['uid']),
            input_stream=data,
            format=yt.JsonFormat(attributes={'encode_utf8': False}),
            raw=False,
        )

    await run_in_executor(_proc, yt_client, table, data)


async def read_users_from_yt(yt_client, table, rows_count):
    def _proc(yt_client, table, rows_count):
        if rows_count:
            table_path = yt.TablePath(table, end_index=rows_count)
        else:
            table_path = yt.TablePath(table)

        users = [row["uid"] for row in yt_client.read_table(table_path)]
        return users

    return await run_in_executor(_proc, yt_client, table, rows_count)


async def create_yt_table(yt_client, table):
    def _proc(yt_client, table):
        fields = [
            {'name': 'uid', 'type': 'int64', 'sort_order': 'ascending'},
        ]
        schema = yson.YsonList(fields)
        schema.attributes['strict'] = True
        yt_client.create(
            type='table',
            path=table,
            recursive=True,
            attributes={'schema': schema},
        )

    await run_in_executor(_proc, yt_client, table)


async def remove_yt_table(yt_client, table):
    def _proc(yt_client, table):
        try:
            yt_client.remove(path=table, force=True)
        except Exception as exc:
            log.exception(f'Got exception while removing YT table: {exc}')

    await run_in_executor(_proc, yt_client, table)


async def check_yt_table(yt_client, table):
    def _proc(yt_client, table):
        if not yt_client.exists(table):
            raise RuntimeError(f'{table} table absent in YT')

    await run_in_executor(_proc, yt_client, table)


@tenacity.retry(
    reraise=True,
    wait=tenacity.wait_random(min=YQL_RETRY_DELAY, max=YQL_RETRY_DELAY * 3),
    stop=tenacity.stop_after_attempt(3))
async def calculate_inactive_users(yql_config, db_users_table, active_users_table, inactive_users_table, tmp_folder):
    def _proc(yql_config, db_users_table, active_users_table, inactive_users_table, tmp_folder):
        query = f'''
            PRAGMA yt.TmpFolder = "{tmp_folder}";
            INSERT INTO `{inactive_users_table}` WITH TRUNCATE
                 SELECT s.uid as uid
                   FROM `{db_users_table}` as s
                   LEFT ONLY JOIN `{active_users_table}` as a
                     ON s.uid = a.uid
               ORDER BY uid;
        '''
        with YqlClient(**yql_config) as yql_client:
            request = yql_client.query(query=query, syntax_version=1)
            request.run()
            request.wait_progress()
            if not request.is_success:
                log.error('YQL request failed: ' + str(request.get_results()))
                raise RuntimeError(f'YQL request failed. Status: {request.status}')

    return await run_in_executor(_proc, yql_config, db_users_table, active_users_table, inactive_users_table, tmp_folder)


@dataclass
class YtFreezingParams:
    max_delay: int = 60 * 60
    table_prefix: str = '//home/mail-logs/core/mdb/freezing'
    yt_config: dict = None
    yql_config: dict = None


@dataclass
class DeactivateUsersParams(TaskParams, YtFreezingParams):
    task_name: str = 'deactivate_users'
    state_ttl: timedelta = timedelta(days=731)
    chunk_size: int = 100000
    max_users_count: int = 250000


async def shard_deactivate_users(params: DeactivateUsersParams, stats):
    date_prefix = datetime. today().strftime('%Y-%m-%d')
    await randomize_start_time(max_delay=params.max_delay)

    yt_client = yt.YtClient(**params.yt_config)
    active_users_table = f'{params.table_prefix}/active_users_dump_{date_prefix}'
    db_users_table = f'{params.table_prefix}/deactivate/db_users_{date_prefix}_{params.shard_id}'
    inactive_users_table = f'{params.table_prefix}/deactivate/inactive_users_{date_prefix}_{params.shard_id}'
    tmp_folder = f'{params.table_prefix}/tmp'

    await check_yt_table(yt_client, active_users_table)

    try:
        await create_yt_table(yt_client, db_users_table)
        async with create_cursor_provider(params, stats) as conn:
            async for users_chunk in get_users(
                conn=conn,
                state_ttl=params.state_ttl,
                chunk_size=params.chunk_size,
            ):
                data = [{'uid': uid} for uid in users_chunk]
                await write_users_to_yt(yt_client, db_users_table, data)
                log.info(f'Written {len(data)} uids to YT')

            await calculate_inactive_users(params.yql_config, db_users_table, active_users_table, inactive_users_table, tmp_folder)
            inactive_users = await read_users_from_yt(yt_client, inactive_users_table, params.max_users_count)
            log.info(f'Read {len(inactive_users)} inactive users from YT')

            for uid in inactive_users:
                await deactivate_user(conn, uid)
    finally:
        await remove_yt_table(yt_client, db_users_table)
        await remove_yt_table(yt_client, inactive_users_table)
