import yt.wrapper as yt
import yt.yson as yson

import random
import asyncio
from dataclasses import dataclass
from .helpers import run_in_executor


async def get_start_id(yt_client, table_name):
    def _proc(yt_client, table_name):
        if yt_client.exists(table_name):
            total = yt_client.row_count(table_name)
            if total > 0:
                for row in yt_client.read_table(yt.TablePath(table_name, start_index=total-1, end_index=total)):
                    return int(row['uid'])
        return 0

    return await run_in_executor(_proc, yt_client, table_name)


async def write_data_to_yt(yt_client, table, data):
    def _proc(yt_client, table, data):
        yt_client.write_table(
            table=yt.TablePath(table, append=True),
            input_stream=data,
            format=yt.JsonFormat(attributes={'encode_utf8': False}),
            raw=False,
        )

    await run_in_executor(_proc, yt_client, table, data)


async def create_yt_table(yt_client, table, fields: dict = dict(), auto_fields: set = set()):
    def _proc(yt_client, table, fields, auto_fields):
        if not yt_client.exists(table):
            fields_list = [{'name': n, 'type': t} for n, t in fields.items()]
            fields_list += [{'name': f, 'type': 'string'} for f in auto_fields if f not in fields.keys()]
            schema = yson.YsonList(fields_list)
            schema.attributes['strict'] = False
            yt_client.create_table(path=table, recursive=True, attributes={'schema': schema})

    await run_in_executor(_proc, yt_client, table, fields, auto_fields)


async def randomize_start_time(max_delay):
    sleep_before_start = random.random() * max_delay
    await asyncio.sleep(sleep_before_start)


@dataclass
class ExportParams:
    from_uid: int = 0
    chunk_size: int = 10000
    max_delay: int = 4 * 60 * 60
    yt_config: dict = None
