import yt.wrapper as yt

from dataclasses import dataclass
from mail.shiva.stages.api.props.logger import get_uid_logger

from .export_helper import create_yt_table, write_data_to_yt
from .cursor_provider import ShardDbCursorProvider, create_cursor_provider

log = get_uid_logger(__name__)

YT_EXPORTED_MAILDB_USERS_TABLE = '//home/mail-logs/core/mdb/users/maildb'
YT_EXPORTED_SHARDDB_USERS_TABLE = '//home/mail-logs/core/mdb/users/sharddb'
USERS_CHUNK_SIZE = 1000000
PG_TIMEOUT = 600


@dataclass
class MailDbUsers:
    uid: int = None
    is_here: bool = None
    is_deleted: bool = None


@dataclass
class ShardDbUsers:
    shard_id: int = None
    uid: int = None
    is_deleted: bool = None


async def get_maildb_users(conn):
    uid = 0
    while True:
        async with conn.cursor(timeout=PG_TIMEOUT) as cur:
            await cur.execute(
                '''
                SELECT uid, is_here, is_deleted
                  FROM mail.users
                 WHERE uid > %(uid)s
                 ORDER BY uid
                 LIMIT %(chunk_size)s
                ''',
                dict(
                    uid=uid,
                    chunk_size=USERS_CHUNK_SIZE,
                )
            )
            chunk = [MailDbUsers(**rec) async for rec in cur]

            if chunk:
                uid = chunk[-1].uid
                yield chunk
            else:
                return


async def get_sharddb_users_with_state(sharddb_conn, is_deleted):
    table_name = 'shards.deleted_users' if is_deleted else 'shards.users'
    uid = 0
    while True:
        async with sharddb_conn.cursor(timeout=PG_TIMEOUT) as cur:
            await cur.execute(
                f'''
                SELECT uid, shard_id
                  FROM {table_name}
                 WHERE uid > %(uid)s
                 ORDER BY uid
                 LIMIT %(chunk_size)s
                ''',
                dict(
                    uid=uid,
                    chunk_size=USERS_CHUNK_SIZE,
                )
            )
            chunk = [ShardDbUsers(**rec, is_deleted=is_deleted) async for rec in cur]

            if chunk:
                uid = chunk[-1].uid
                yield chunk
            else:
                return


async def get_sharddb_users(sharddb_conn):
    async for chunk in get_sharddb_users_with_state(sharddb_conn, is_deleted=True):
        yield chunk
    async for chunk in get_sharddb_users_with_state(sharddb_conn, is_deleted=False):
        yield chunk


async def export_maildb_users(params, stats):
    yt_client = yt.YtClient(**params.settings.yt.yt_config)
    yt_table_name = f'{YT_EXPORTED_MAILDB_USERS_TABLE}/{params.shard_id}'
    fields = {
        'shard_id': 'uint64',
        'uid': 'uint64',
        'is_here': 'boolean',
        'is_deleted': 'boolean',
    }
    await create_yt_table(yt_client, yt_table_name, fields=fields)

    async with create_cursor_provider(params, stats) as conn:
        async for users in get_maildb_users(conn):
            data = [{
                'shard_id': params.shard_id,
                'uid': usr.uid,
                'is_here': usr.is_here,
                'is_deleted': usr.is_deleted,
            } for usr in users]
            await write_data_to_yt(yt_client, yt_table_name, data)


async def export_sharddb_users(params, stats):
    yt_client = yt.YtClient(**params.settings.yt.yt_config)
    yt_table_name = YT_EXPORTED_SHARDDB_USERS_TABLE
    fields = {
        'shard_id': 'uint64',
        'uid': 'uint64',
        'is_deleted': 'boolean',
    }
    await create_yt_table(yt_client, yt_table_name, fields=fields)

    sharddb_conn = ShardDbCursorProvider(params.sharpei, stats)
    try:
        async for users in get_sharddb_users(sharddb_conn):
            data = [{
                'shard_id': usr.shard_id,
                'uid': usr.uid,
                'is_deleted': usr.is_deleted,
            } for usr in users]
            await write_data_to_yt(yt_client, yt_table_name, data)
    finally:
        sharddb_conn.close()
