from dataclasses import dataclass

from mail.shiva.stages.api.props.logger import get_state_history_logger

state_history_log = get_state_history_logger()


@dataclass
class UserState:
    uid: int = None
    state: str = None
    is_here: bool = None
    is_deleted: bool = None
    notifies_count: int = None


def log_update_user_state(user: UserState, state, notifies_count):
    state_history_log.info(
        msg='Updating user state',
        uid=user.uid,
        old_state=user.state,
        new_state=state,
        old_notifies_count=user.notifies_count,
        new_notifies_count=notifies_count,
    )


async def db_update_user_state(cursor, user: UserState, state):
    await cursor.execute(
        '''SELECT * FROM code.update_user_state(%(uid)s, %(state)s)''',
        dict(
            uid=user.uid,
            state=state,
        )
    )
    log_update_user_state(user, state, user.notifies_count)


async def db_update_user_state_with_notifies_count(cursor, user: UserState, state, notifies_count):
    await cursor.execute(
        '''SELECT * FROM code.update_user_state(%(uid)s, %(state)s, %(notifies_count)s::smallint)''',
        dict(
            uid=user.uid,
            state=state,
            notifies_count=notifies_count,
        )
    )
    log_update_user_state(user, state, notifies_count)


async def db_get_user_state(cursor, uid):
    await cursor.execute(
        '''
            SELECT uid, state, is_here, is_deleted, notifies_count
              FROM mail.users
             WHERE uid = %(uid)s
        ''',
        dict(uid=uid)
    )
    res = await cursor.fetchone()
    return res and UserState(**res)


def is_live_user(user: UserState):
    return user is not None and user.is_here and not user.is_deleted
