import asyncio
import aiopg
import psycopg2
from psycopg2.extras import RealDictCursor
from dataclasses import dataclass

from mail.shiva.stages.api.props.services.sharpei import get_shard_dsn
from .task import TaskParams
from mail.shiva.stages.api.props.logger import get_uid_logger

log = get_uid_logger(__name__)
INIT_POP3_TIMEOUT = 600


@dataclass
class UidFid:
    uid: str = None
    fid: str = None


def format_error(error):
    return '%s %s' % (type(error), str(error).replace('\n', '\\n'))


async def get_folders(conn, limit):
    async with conn.cursor() as cur:
        await cur.execute(
            '''
            SELECT uid, fid
              FROM mail.folders
             WHERE (pop3state).enabled = true
               AND (pop3state).initialized = false
             LIMIT %(limit)s
            ''',
            dict(limit=limit)
        )
        return [UidFid(**v) async for v in cur]


async def is_here(conn, uid):
    async with conn.cursor() as cur:
        await cur.execute(
            '''
            SELECT *
              FROM mail.users
             WHERE uid = %(uid)s
               AND is_here
               AND NOT is_deleted
            ''',
            dict(uid=uid)
        )
        return await cur.fetchone() is not None


async def init_pop3_folder(conn, uid, fid):
    if not await is_here(conn, uid):
        log.info('user is not here', uid=uid)
        return False
    async with conn.cursor(timeout=INIT_POP3_TIMEOUT) as cur:
        try:
            await cur.callproc('code.initialize_pop3_folder', (uid, fid))
            res = await cur.fetchone()
            log.info(f'init status=ok fid={fid} revision={res["initialize_pop3_folder"]}', uid=uid)
            return True
        except (psycopg2.Error, asyncio.TimeoutError) as error:
            log.error(f'init status=fail fid={fid} reason={format_error(error)}', uid=uid)
            return False


@dataclass
class InitPop3FolderParams(TaskParams):
    task_name: str = 'init_pop3_folders'
    max_count: int = 10000


async def shard_init_pop3_folder(params: InitPop3FolderParams, stats):
    async with aiopg.connect(await get_shard_dsn(params.sharpei, params.db_user, params.shard_id, stats), cursor_factory=RealDictCursor) as conn:
        folders = await get_folders(conn, params.max_count)
        ok = 0
        fail = 0
        for folder in folders:
            if await init_pop3_folder(conn, folder.uid, folder.fid):
                ok += 1
            else:
                fail += 1
        log.info('init_pop3_folder finished: processed=%s succeed=%s failed=%s', ok + fail, ok, fail)
