import json

import asyncio
import aiohttp
from aiohttp.client_exceptions import ClientConnectorError
from typing import List, Optional, Union
from dataclasses import dataclass
from collections import defaultdict
from enum import Enum

from mail.shiva.stages.api.settings.blackbox import BlackboxSettings
from mail.shiva.stages.api.settings.sendr import SendrSettings
from mail.shiva.stages.api.settings.tvm import TvmSettings
from mail.python.theatre.detail.tvm import TvmServiceTickets
from mail.shiva.stages.api.props.logger import get_uid_logger
from mail.python.theatre.profiling.http import ProfiledClientSession
from mail.shiva.stages.api.settings.log import http_logger

log = get_uid_logger(__name__)
SUBSCRIPTION_MAIL_STATUS_ATTR = '203'
MAIL_STATUS_FROZEN = '2'


class BbSids(Enum):
    MAIL = '2'
    DIRECT = '14'

    @property
    def db_field(self):
        return f'subscription.suid.{self.value}'


def _is_recovery_address(a: dict) -> bool:
    return not a['rpop'] and not a['native'] and not a['unsafe'] and a['validated'] and not a['silent']


def _is_default_address(a: dict) -> bool:
    return a['default']


class Metrics(object):
    def __init__(self, stats):
        self.logger = log
        self._stats = stats

    def cannot_resolve_users(self, users):
        self.increase('cannot_resolve_users')

    def successfully_sended(self, uid, task_id):
        self.increase('successfully_sended')
        self.logger.info(f'async send mail with task_id={task_id}', uid=uid)

    def successfully_sended_with_strange_response(self, uid, resp):
        self.increase('successfully_sended_with_strange_response')
        self.logger.warning(f'async send mail with strange response: {resp}', uid=uid)

    def missing_uid(self, uid):
        self.increase('missing_uid')
        self.logger.info('there are no such uid', uid=uid)

    def missing_sid2(self, uid):
        self.increase('missing_sid2')
        self.logger.info('there are no sid=2, probably nonmail user', uid=uid)

    def frozen_user(self, uid):
        self.increase('frozen_user')
        self.logger.info('got frozen user', uid=uid)

    def missing_email(self, uid):
        self.increase('missing_email')
        self.logger.info('there are no emails, probably nonmail user', uid=uid)

    def blackbox_error(self, msg):
        self.increase('blackbox_error')
        self.logger.error(msg)

    def sendr_error(self, uid, resp):
        self.increase('sendr_error')
        self.logger.error(f'strange sendr response {resp}', uid=uid)

    def sendr_exception(self, uid, exc):
        self.increase('sendr_exception')
        self.logger.exception(f'catch exception during sendr request: {exc}', uid=uid)

    def increase(self, name):
        self._stats.increase_task_meter(f"{name}")

    def increase_global_meter(self, name):
        self._stats.increase_global_meter(f"{name}")

    def put_in_hist(self, name: str, value: Union[float, int] = 1):
        self._stats.put_in_hist(name, value)


@dataclass
class User(object):
    uid: str = ''
    default_address: Optional[str] = None
    addresses: List[str] = None
    lang: str = ''
    first_name: str = ''
    is_direct: bool = False


def _parse_recovery_addresses(u: List[dict]) -> List[str]:
    return list(
        map(
            lambda a: a['address'],
            filter(
                _is_recovery_address, u
            )
        )
    )


def _parse_default_addresses(u: List[dict]) -> Optional[str]:
    default = list(
        map(
            lambda a: a['address'],
            filter(
                _is_default_address, u
            )
        )
    )

    return default[0] if len(default) else None


def _parse_blackbox(users: List[dict], metrics: Metrics) -> List[User]:
    ret = []

    for u in users:
        try:
            if 'address-list' not in u or len(u['address-list']) == 0:
                metrics.missing_email(uid=u["id"])
                continue

            if 'value' not in u['uid']:
                metrics.missing_uid(uid=u["id"])
                continue

            if u['attributes'].get(SUBSCRIPTION_MAIL_STATUS_ATTR) == MAIL_STATUS_FROZEN:
                metrics.frozen_user(uid=u["id"])
                continue

            if u['dbfields'][BbSids.MAIL.db_field] == '':
                metrics.missing_sid2(uid=u["id"])
                continue

            default = _parse_default_addresses(u['address-list'])
            recovery = _parse_recovery_addresses(u['address-list'])
            is_direct = bool(u['dbfields'].get(BbSids.DIRECT.db_field))

            if default is not None and default not in recovery:
                recovery.append(default)

            lang = u['dbfields']['userinfo.lang.uid']
            first_name = u['dbfields']['userinfo.firstname.uid'] or default

            ret.append(User(
                uid=u["id"],
                addresses=recovery,
                lang=lang,
                default_address=default,
                first_name=first_name,
                is_direct=is_direct,
            ))
        except:
            metrics.blackbox_error(f'strange BB user format: {u}')

    return ret


class Templates(object):
    def __init__(self, default: str = '', **kwargs):
        self.__templates = defaultdict(lambda: default, **kwargs)

    def __getitem__(self, item):
        return self.__templates[item]


async def sendr(cfg: SendrSettings, user: User, client, metrics: Metrics, templates: Templates, recovery_send: bool = True) -> bool:
    auth = aiohttp.BasicAuth(login=cfg.login)

    data = {
        'to': [
            {'email': address} for address in (user.addresses if recovery_send else [user.default_address])
        ],
        'async': True,
        'args': {'name': user.first_name, 'email': user.default_address}
    }

    for i in range(cfg.tries):
        try:
            async with client.post(url=cfg.location(campaign_id=templates[user.lang]), auth=auth, json=data) as resp:
                if resp.status == 200:
                    resp = await resp.text()
                    try:
                        js_resp = json.loads(resp)
                        metrics.successfully_sended(user.uid, js_resp['result']['task_id'])
                    except Exception:
                        metrics.successfully_sended_with_strange_response(user.uid, resp)
                    return True
                elif resp.status / 100 == 4:
                    metrics.sendr_error(uid=user.uid, resp=await resp.text())
                    return False
                else:
                    metrics.sendr_error(uid=user.uid, resp=await resp.text())
                    continue
        except (asyncio.TimeoutError, ClientConnectorError) as exc:
            metrics.sendr_exception(uid=user.uid, exc=exc)

    metrics.sendr_error(uid=user.uid, resp=None)

    return False


async def _blackbox(cfg: BlackboxSettings, req_params: dict, bb_tvm_ticket: str, client, metrics, parser) -> List[User]:
    params = {
        'method': 'userinfo',
        'userip': '127.0.0.1',
        'sid': '2',
        'format': 'json',
        'emails': 'getall',
        'dbfields': ','.join(['userinfo.lang.uid', 'userinfo.firstname.uid',
                              BbSids.MAIL.db_field, BbSids.DIRECT.db_field]),
        'attributes': SUBSCRIPTION_MAIL_STATUS_ATTR
    }
    params.update(req_params)

    headers = {
        'x-ya-service-ticket': bb_tvm_ticket
    }

    for i in range(cfg.tries):
        try:
            async with client.get(url=cfg.location, params=params, headers=headers) as resp:
                if resp.status != 200:
                    metrics.blackbox_error(msg=f'strange bb response status: {resp.status}')
                else:
                    js = json.loads(await resp.text())

                    if 'users' in js:
                        return parser(users=js['users'], metrics=metrics)
                    else:
                        metrics.blackbox_error(msg=f'there are no "users" in response: {js}')
        except (asyncio.TimeoutError, ClientConnectorError) as exc:
            metrics.blackbox_error(msg=f'catch exception during blackbox request: {exc}')

    metrics.blackbox_error('cannot get valid bb response: try limit')

    return []


async def blackbox(cfg: BlackboxSettings, uids: List[int], bb_tvm_ticket: str, client, metrics, parser=_parse_blackbox) -> List[User]:
    req_params = {
        'uid': ','.join(map(lambda x: str(x), uids)),
    }

    return await _blackbox(cfg, req_params, bb_tvm_ticket, client, metrics, parser)


async def blackbox_by_email(cfg: BlackboxSettings, email: str, bb_tvm_ticket: str, client, metrics, parser=_parse_blackbox) -> List[User]:
    req_params = {
        'login': email,
        'sid': 'smtp',
    }

    return await _blackbox(cfg, req_params, bb_tvm_ticket, client, metrics, parser)


@dataclass
class NotifyParams:
    bb_settings: BlackboxSettings = None
    sendr_settings: SendrSettings = None
    tvm_ids: TvmSettings = None
    tvm_tickets: TvmServiceTickets = None


async def _is_active_bb_user(bb_cfg: BlackboxSettings, uid: int, client, bb_tvm_ticket, metrics, parser=_parse_blackbox):
    users = await blackbox(cfg=bb_cfg, uids=[uid], bb_tvm_ticket=bb_tvm_ticket, client=client, metrics=metrics, parser=parser)
    if len(users) != 1:
        return False
    return True


async def is_active_bb_user(params: NotifyParams, uid: int, metrics):
    bb_tvm_ticket = await params.tvm_tickets.get(params.tvm_ids.bb_id)
    async with ProfiledClientSession(metrics=metrics, logger=http_logger.get_logger(), timeout=params.bb_settings.timeout, json_serialize=json.dumps) as client:
        return await _is_active_bb_user(params.bb_settings, uid, client, bb_tvm_ticket, metrics)


async def _is_valid_default_address(bb_cfg: BlackboxSettings, user: User, client, bb_tvm_ticket, metrics, parser=_parse_blackbox):
    if user.default_address is None:
        return False
    users_by_email = await blackbox_by_email(bb_cfg, user.default_address, bb_tvm_ticket, client, metrics, parser)
    if len(users_by_email) != 1 or users_by_email[0].uid != user.uid:
        return False
    return True


async def is_valid_default_address(params: NotifyParams, user: User, metrics):
    bb_tvm_ticket = await params.tvm_tickets.get(params.tvm_ids.bb_id)
    async with ProfiledClientSession(metrics=metrics, logger=http_logger.get_logger(), timeout=params.bb_settings.timeout, json_serialize=json.dumps) as client:
        return await _is_valid_default_address(params.bb_settings, user, client, bb_tvm_ticket, metrics)


async def _quick_send(bb_cfg: BlackboxSettings, sendr_cfg: SendrSettings, client, bb_tvm_ticket, uid: int,
                      templates: Templates, metrics, recovery_send: bool = True, parser=_parse_blackbox) -> bool:
    users = await blackbox(cfg=bb_cfg, uids=[uid], bb_tvm_ticket=bb_tvm_ticket, client=client, metrics=metrics,
                           parser=parser)
    if len(users) != 1:
        return False
    if not await _is_valid_default_address(bb_cfg=bb_cfg, user=users[0], client=client,
                                           bb_tvm_ticket=bb_tvm_ticket, metrics=metrics, parser=parser):
        return False
    return await sendr(cfg=sendr_cfg, user=users[0], client=client, metrics=metrics, templates=templates,
                       recovery_send=recovery_send)


async def quick_send(params: NotifyParams, uid: int, templates: Templates, metrics, recovery_send: bool = True,
                     parser=_parse_blackbox) -> bool:
    bb_tvm_ticket = await params.tvm_tickets.get(params.tvm_ids.bb_id)
    async with ProfiledClientSession(meters=metrics, logger=http_logger.get_logger(), timeout=params.bb_settings.timeout, json_serialize=json.dumps) as client:
        return await _quick_send(params.bb_settings, params.sendr_settings, client, bb_tvm_ticket, uid, templates,
                                 metrics, recovery_send, parser)
